/**
 * The type of the activity displayed as a status message with its name (e.g.
 * *Playing \<game\>* or *Listening to \<music\>*).
 */
export declare const enum RPCActivity {
    Game = 0,
    Listening = 1,
    Streaming = 2,
    Watching = 3,
    Custom = 4,
    Competing = 5
}
/** Static, hard-coded event objects sent by transports at certain conditions.*/
export declare const staticEvents: Readonly<{
    /**
     * A fake, hard-coded Discord event to spoof the presence of
     * official Discord client (which makes browser to actually start a
     * communication with the DisConnection).
     */
    readonly ready: {
        cmd: "DISPATCH";
        data: {
            v: 1;
            config: null;
            user: null;
        };
        evt: "READY";
        nonce: null;
    };
}>;
/**
 * Known *identifiers* of packets, which tends to be either a `code` of
 * `cmd`, `events` (`evt`) or `type` (for `DEEP_LINK`).
 */
export declare const knownPacketID: Readonly<{
    codes: readonly ["INVITE_BROWSER", "GUILD_TEMPLATE_BROWSER", "AUTHORIZE", "DEEP_LINK", "SET_ACTIVITY"];
    types: readonly ["CHANNEL"];
    events: readonly ["READY", "ERROR"];
}>;
/** Alias type for single element of `knownMsgEl.codes`. */
export type code = typeof knownPacketID.codes[number];
/** Alias type for single element of `knownMsgEl.types`. */
export type type = typeof knownPacketID.types[number];
interface UnknownPacket {
    /** Message type/command. */
    cmd: string;
    /** Nonce identifying the communication. */
    nonce: string | null;
}
/** A generic non-precise type of Discord's incoming messages. */
export interface UnknownMessage extends UnknownPacket {
    /** Message arguments. */
    args: Record<string, unknown>;
}
/** An object type of known Discord's incoming messages. */
export interface Message<C extends code, T extends string> extends UnknownMessage {
    cmd: C;
    args: C extends `${"INVITE" | "GUILD_TEMPLATE"}_BROWSER` ? {
        /** An invitation code. */
        code: string;
    } : C extends "AUTHORIZE" ? {
        /** An array of OAuth2 scopes. */
        scopes: (`applications.${`builds.${"read" | "upload"}` | `commands.${"" | "permissions."}update` | "entitlements" | "store.update"}` | "bot" | "connections" | "dm_channels.read" | "email" | "gdm.join" | "guilds" | `guilds.${"join" | "members.read"}` | "identify" | `${"activities" | "messages" | "relationships" | `rpc.${"notifications" | "voice"}`}.read` | `${"activities" | `rpc.${"activities" | "voice"}`}.write` | "voice" | "webhook.incoming")[];
        /** An application's client_id. */
        client_id: string;
        /** One-time use RPC token. */
        rpc_token?: string;
        /** A username of guest account to create if the user does not have Discord. */
        username?: string;
    } : C extends "DEEP_LINK" ? {
        type: T extends string ? T : string;
        params: T extends "CHANNEL" ? {
            guildId: string;
            channelId?: string;
            search: string;
            fingerprint: string;
        } : Record<string, unknown>;
    } : C extends "SET_ACTIVITY" ? {
        /** The application's process id. */
        pid: number;
        activity: {
            /** The activity name. */
            name: string;
            /** The activity type, one of {@link RPCActivity}. */
            type: RPCActivity;
            /** A stream URL, validated only when `type` is {@link RPCActivity.Listening}. */
            url?: string;
            /** A unix timestamp (in milliseconds) of when the activity was added to the user's session. */
            created_at: number;
            /** Unix timestamps for start and/or end of the activity. */
            timestamps?: Partial<Record<"start" | "end", number>>;
            /** Application ID for the activity. */
            application_id?: number;
            /** What the user is currently doing as a part of given activity. */
            details: string;
            /** The user's current party status. */
            state?: string;
            /** An emoji used for a custom status. */
            emoji?: {
                name?: string;
                id?: number;
                animated?: string;
            };
            /** An information about the current party participating in the user's activity. */
            party?: {
                id?: number;
                size?: [current_size: number, max_size: number];
            };
            /** Images for the presence and their hover texts. */
            assets?: Partial<Record<`${"large" | "small"}_${"image" | "text"}`, string>>;
            /** Secrets for Rich Presence joining and spectating. */
            secrets?: Partial<Record<"join" | "spectate" | "match", string>>;
            /** Whether or not the activity is an instanced game session. */
            instance?: boolean;
            /**
             * An integer in range 0-511 (unsigned, 9 bits) which identifies flags.
             *
             * @summary
             *
             * Bits in number has specified flag as seen in below table:
             *
             * |  BIT | FLAG                        |
             * | ---- | --------------------------- |
             * | 1    |`INSTANCE`                   |
             * | 2    |`JOIN`                       |
             * | 3    |`SPECTATE`                   |
             * | 4    |`JOIN_REQUEST`               |
             * | 5    |`SYNC`                       |
             * | 6    |`PLAY`                       |
             * | 7    |`PARTY_PRIVACY_FRIENDS`      |
             * | 8    |`PARTY_PRIVACY_VOICE_CHANNEL`|
             * | 9    |`EMBEDDED`                   |
             */
            flags?: number;
            /** An array of buttons shown in the Rich Presence. */
            buttons?: {
                label: string;
                url: string;
            }[] & {
                length: 0 | 1 | 2;
            };
        };
    } : never;
}
/**
 * Generic response checker, assumes Discord will do requests of certain type
 * based on `cmd` and `argsType` values.
 */
export declare function isMessage<C extends code | undefined = undefined, T extends type | undefined = undefined>(data: unknown, cmd?: C | C[], argsType?: T): data is C extends string ? Message<C, T extends string ? T : never> : UnknownMessage;
export declare function messageDefaultResponse(message: UnknownMessage): Readonly<{
    readonly nonce: string | null;
    readonly evt?: null;
    readonly cmd: string;
    readonly data: {
        guildTemplate?: {
            code: {} | null;
        };
        code: {} | null;
    } | null;
}>;
export {};
