import { BaseExtension } from '../../extension';
export interface chatCompletionRequestMessage {
    role: 'system' | 'user' | 'assistant' | 'tool';
    content: string | null | Content[];
    reasoning?: string | null;
    reasoning_content?: string | null;
    name?: string;
    tool_calls?: any[];
}
export interface Content {
    type: 'text' | 'image_url' | 'input_audio';
    text?: string;
    image_url?: string;
    input_audio?: InputAudio;
}
export interface InputAudio {
    data: string;
    format: 'mp3' | 'wav' | 'ogg' | 'flac';
}
export interface ToolFunction {
    name: string;
    description?: string;
    parameters?: Record<string, unknown>;
    strict?: boolean | null;
}
export interface Tool {
    type: 'function';
    function: ToolFunction;
}
export interface ToolCallOptions {
    tools?: Tool[];
}
export interface ToolCallSpec {
    type: 'function';
    function: {
        name: string;
    };
}
export type ToolChoice = 'none' | 'auto' | 'required' | ToolCallSpec;
export interface chatCompletionRequest {
    model: string;
    messages: chatCompletionRequestMessage[];
    thread_id?: string;
    return_progress?: boolean;
    tools?: Tool[];
    tool_choice?: ToolChoice;
    temperature?: number | null;
    dynatemp_range?: number | null;
    dynatemp_exponent?: number | null;
    top_k?: number | null;
    top_p?: number | null;
    min_p?: number | null;
    typical_p?: number | null;
    repeat_penalty?: number | null;
    repeat_last_n?: number | null;
    presence_penalty?: number | null;
    frequency_penalty?: number | null;
    dry_multiplier?: number | null;
    dry_base?: number | null;
    dry_allowed_length?: number | null;
    dry_penalty_last_n?: number | null;
    dry_sequence_breakers?: string[] | null;
    xtc_probability?: number | null;
    xtc_threshold?: number | null;
    mirostat?: number | null;
    mirostat_tau?: number | null;
    mirostat_eta?: number | null;
    n_predict?: number | null;
    n_indent?: number | null;
    n_keep?: number | null;
    stream?: boolean | null;
    stop?: string | string[] | null;
    seed?: number | null;
    logit_bias?: {
        [key: string]: number;
    } | null;
    n_probs?: number | null;
    min_keep?: number | null;
    t_max_predict_ms?: number | null;
    image_data?: Array<{
        data: string;
        id: number;
    }> | null;
    id_slot?: number | null;
    cache_prompt?: boolean | null;
    return_tokens?: boolean | null;
    samplers?: string[] | null;
    timings_per_token?: boolean | null;
    post_sampling_probs?: boolean | null;
    chat_template_kwargs?: chat_template_kdict | null;
}
export interface chat_template_kdict {
    enable_thinking: false;
}
export interface chatCompletionChunkChoiceDelta {
    content?: string | null;
    role?: 'system' | 'user' | 'assistant' | 'tool';
    tool_calls?: any[];
}
export interface chatCompletionChunkChoice {
    index: number;
    delta: chatCompletionChunkChoiceDelta;
    finish_reason?: 'stop' | 'length' | 'tool_calls' | 'content_filter' | 'function_call' | null;
}
export interface chatCompletionPromptProgress {
    cache: number;
    processed: number;
    time_ms: number;
    total: number;
}
export interface chatCompletionChunk {
    id: string;
    object: 'chat.completion.chunk';
    created: number;
    model: string;
    choices: chatCompletionChunkChoice[];
    system_fingerprint?: string;
    prompt_progress?: chatCompletionPromptProgress;
}
export interface chatCompletionChoice {
    index: number;
    message: chatCompletionRequestMessage;
    finish_reason: 'stop' | 'length' | 'tool_calls' | 'content_filter' | 'function_call';
    logprobs?: any;
}
export interface chatCompletion {
    id: string;
    object: 'chat.completion';
    created: number;
    model: string;
    choices: chatCompletionChoice[];
    usage?: {
        prompt_tokens: number;
        completion_tokens: number;
        total_tokens: number;
    };
    system_fingerprint?: string;
}
export interface modelInfo {
    id: string;
    name: string;
    quant_type?: string;
    providerId: string;
    port: number;
    sizeBytes: number;
    tags?: string[];
    path?: string;
    embedding?: boolean;
    [key: string]: any;
}
export type listResult = modelInfo[];
export interface SessionInfo {
    pid: number;
    port: number;
    model_id: string;
    model_path: string;
    is_embedding: boolean;
    api_key: string;
    mmproj_path?: string;
}
export interface UnloadResult {
    success: boolean;
    error?: string;
}
export interface chatOptions {
    providerId: string;
    sessionId: string;
    /** Full OpenAI ChatCompletionRequest payload */
    payload: chatCompletionRequest;
}
export interface ImportOptions {
    modelPath: string;
    mmprojPath?: string;
    modelSha256?: string;
    modelSize?: number;
    mmprojSha256?: string;
    mmprojSize?: number;
}
export interface importResult {
    success: boolean;
    modelInfo?: modelInfo;
    error?: string;
}
/**
 * Base AIEngine
 * Applicable to all AI Engines
 */
export declare abstract class AIEngine extends BaseExtension {
    abstract readonly provider: string;
    /**
     * On extension load, subscribe to events.
     */
    onLoad(): void;
    /**
     * Registers AI Engines
     */
    registerEngine(): void;
    /**
     * Gets model info
     * @param modelId
     */
    abstract get(modelId: string): Promise<modelInfo | undefined>;
    /**
     * Lists available models
     */
    abstract list(): Promise<modelInfo[]>;
    /**
     * Loads a model into memory
     */
    abstract load(modelId: string, settings?: any): Promise<SessionInfo>;
    /**
     * Unloads a model from memory
     */
    abstract unload(sessionId: string): Promise<UnloadResult>;
    /**
     * Sends a chat request to the model
     */
    abstract chat(opts: chatCompletionRequest, abortController?: AbortController): Promise<chatCompletion | AsyncIterable<chatCompletionChunk>>;
    /**
     * Deletes a model
     */
    abstract delete(modelId: string): Promise<void>;
    /**
     * Updates a model
     */
    abstract update(modelId: string, model: Partial<modelInfo>): Promise<void>;
    /**
     * Imports a model
     */
    abstract import(modelId: string, opts: ImportOptions): Promise<void>;
    /**
     * Aborts an ongoing model import
     */
    abstract abortImport(modelId: string): Promise<void>;
    /**
     * Get currently loaded models
     */
    abstract getLoadedModels(): Promise<string[]>;
    /**
     * Check if a tool is supported by the model
     * @param modelId
     */
    abstract isToolSupported(modelId: string): Promise<boolean>;
}
