"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerForecastRoutes = registerForecastRoutes;
var _lodash = require("lodash");
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _forecastHelpers = require("./utils/forecastHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerForecastRoutes(apiRouter, forecastService) {
  // create forecaster
  apiRouter.post('/forecasters', forecastService.putForecaster);
  apiRouter.post('/forecasters/{dataSourceId}', forecastService.putForecaster);

  // put forecaster
  apiRouter.put('/forecasters/{forecasterId}', forecastService.putForecaster);
  apiRouter.put('/forecasters/{forecasterId}/{dataSourceId}', forecastService.putForecaster);

  // FIXME: routes not used in the UI, therefore no data source id
  apiRouter.post('/forecasters/_search', forecastService.searchForecaster);

  /**
   * Search forecast results routes
   * 
   * We use 'by-source' and 'by-index' path segments to avoid route conflicts between
   * paths with different parameter types. Without these segments, routes like:
   *   /forecasters/results/_search/{resultIndex}
   *   /forecasters/results/_search/{dataSourceId}
   * would conflict because OpenSearch can't distinguish between parameter types in the same position.
   * 
   * Current route structure:
   * 1. Search by source (no params)     : /by-source/_search
   * 2. Search by source with ID         : /by-source/{dataSourceId}/_search
   * 3. Search by index pattern          : /by-index/{resultIndex}/_search
   * 4. Search by both index and source  : /by-index/{resultIndex}/by-source/{dataSourceId}/_search
   */

  // Search with no parameters
  apiRouter.post('/forecasters/results/by-source/_search', forecastService.searchResults);

  // Search by data source ID
  apiRouter.post('/forecasters/results/by-source/{dataSourceId}/_search', forecastService.searchResults);

  // Search by result index pattern
  apiRouter.post('/forecasters/results/by-index/{resultIndex}/_search', forecastService.searchResults);

  // Search by both result index and data source ID
  apiRouter.post('/forecasters/results/by-index/{resultIndex}/by-source/{dataSourceId}/_search', forecastService.searchResults);

  // list forecasters
  apiRouter.get('/forecasters/_list', forecastService.getForecasters);
  apiRouter.get('/forecasters/_list/{dataSourceId}', forecastService.getForecasters);

  // run once forecaster
  apiRouter.post('/forecasters/{forecasterId}/_run_once', forecastService.runOnceForecaster);
  apiRouter.post('/forecasters/{forecasterId}/_run_once/{dataSourceId}', forecastService.runOnceForecaster);

  // get forecaster forecast results
  apiRouter.get('/forecasters/{id}/results/{isRunOnce}/{resultIndex}', forecastService.getForecastResults);
  apiRouter.get('/forecasters/{id}/results/{isRunOnce}/{resultIndex}/{dataSourceId}', forecastService.getForecastResults);

  // delete forecaster
  apiRouter.delete('/forecasters/{forecasterId}', forecastService.deleteForecaster);
  apiRouter.delete('/forecasters/{forecasterId}/{dataSourceId}', forecastService.deleteForecaster);

  // start forecaster
  apiRouter.post('/forecasters/{forecasterId}/start', forecastService.startForecaster);
  apiRouter.post('/forecasters/{forecasterId}/start/{dataSourceId}', forecastService.startForecaster);

  // stop forecaster
  apiRouter.post('/forecasters/{forecasterId}/stop', forecastService.stopForecaster);
  apiRouter.post('/forecasters/{forecasterId}/stop/{dataSourceId}', forecastService.stopForecaster);
  apiRouter.get('/forecasters/{forecasterId}/_profile', forecastService.getForecasterProfile);

  // get forecaster
  apiRouter.get('/forecasters/{forecasterId}', forecastService.getForecaster);
  apiRouter.get('/forecasters/{forecasterId}/{dataSourceId}', forecastService.getForecaster);

  // match forecaster
  apiRouter.get('/forecasters/{forecasterName}/_match', forecastService.matchForecaster);
  apiRouter.get('/forecasters/{forecasterName}/_match/{dataSourceId}', forecastService.matchForecaster);

  // get forecaster count
  apiRouter.get('/forecasters/_count', forecastService.getForecasterCount);
  apiRouter.get('/forecasters/_count/{dataSourceId}', forecastService.getForecasterCount);

  // post get top forecast results
  apiRouter.post('/forecasters/{forecasterId}/_topForecasts/{isRunOnce}', forecastService.getTopForecastResults);
  apiRouter.post('/forecasters/{forecasterId}/_topForecasts/{isRunOnce}/{dataSourceId}', forecastService.getTopForecastResults);

  // validate forecaster
  apiRouter.post('/forecasters/_validate/{validationType}', forecastService.validateForecaster);
  apiRouter.post('/forecasters/_validate/{validationType}/{dataSourceId}', forecastService.validateForecaster);

  // suggest forecaster
  apiRouter.post('/forecasters/_suggest/{suggestType}', forecastService.suggestForecaster);
  apiRouter.post('/forecasters/_suggest/{suggestType}/{dataSourceId}', forecastService.suggestForecaster);
}
class ForecastService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "deleteForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        // dataSourceId will be "" and fall back to use the existing client for local cluster
        // On the other hand, in MDS world, the open search legacy client (this.client) will
        // be undefined and it will pickup the data source client 
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.deleteForecaster', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - deleteForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "runOnceForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId = '',
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.runOnceForecaster', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            // return taskId
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - runOnceForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "putForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;

        //@ts-ignore
        const ifSeqNo = request.body.seqNo;
        //@ts-ignore
        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        let params = {
          forecasterId: forecasterId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await callWithRequest('forecast.updateForecaster', params);
        } else {
          response = await callWithRequest('forecast.createForecaster', {
            body: params.body
          });
        }
        const resp = {
          ...response.forecaster,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _forecastHelpers.convertForecastKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Forecast - PutForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "validateForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        const response = await callWithRequest('forecast.validateForecaster', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - validateForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "suggestForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          suggestType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        const response = await callWithRequest('forecast.suggestForecaster', {
          body: requestBody,
          suggestType: suggestType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - suggestForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const forecasterResponse = await callWithRequest('forecast.getForecaster', {
          forecasterId
        });

        // Populating static forecaster fields
        const staticFields = {
          id: forecasterResponse._id,
          primaryTerm: forecasterResponse._primary_term,
          seqNo: forecasterResponse._seq_no,
          // the backend returns a response with forecaster field.
          ...(0, _forecastHelpers.convertStaticFieldsToCamelCase)(forecasterResponse.forecaster)
        };

        // Get real-time and run-once task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no forecaster jobs have been ran on a new cluster.
        // let realtimeTasksResponse = {} as any;
        // let runOnceTasksResponse = {} as any;
        // try {
        //   const callWithRequest = getClientBasedOnDataSource(
        //     context,
        //     this.dataSourceEnabled,
        //     request,
        //     dataSourceId,
        //     this.client
        //   );

        //   realtimeTasksResponse = await callWithRequest('forecast.searchTasks', {
        //     body: getLatestTaskForForecasterQuery(forecasterId, true),
        //   });

        //   runOnceTasksResponse = await callWithRequest('forecast.searchTasks', {
        //     body: getLatestTaskForForecasterQuery(forecasterId, false),
        //   });
        // } catch (err) {
        //   if (!isIndexNotFoundError(err)) {
        //     throw err;
        //   }
        // }

        // const realtimeTask = get(
        //   get(realtimeTasksResponse, 'hits.hits', []).map((taskResponse: any) => {
        //     return {
        //       id: get(taskResponse, '_id'),
        //       ...get(taskResponse, '_source'),
        //     };
        //   }),
        //   0
        // );
        // const runOnceTask = get(
        //   get(runOnceTasksResponse, 'hits.hits', []).map(
        //     (taskResponse: any) => {
        //       return {
        //         id: get(taskResponse, '_id'),
        //         ...get(taskResponse, '_source'),
        //       };
        //     }
        //   ),
        //   0
        // );

        const taskAndJobFields = (0, _forecastHelpers.convertTaskAndJobFieldsToCamelCase)(forecasterResponse.realtime_task, forecasterResponse.run_once_task, forecasterResponse.forecaster_job);

        // Combine the static and task-and-job-related fields into
        // a final response
        const finalResponse = {
          ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        // if the forecaster is not found (e.g., deleted while on the detail page), return an empty response
        // this is to avoid the error message from the frontend where the forecaster is not found
        // the error is triggered by useEffect of useFetchForecasterInfo in ForecasterDetail
        if (err.statusCode === 404) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {}
            }
          });
        }
        console.log('Forecast - Unable to get forecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        //@ts-ignore
        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime;
        //@ts-ignore
        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          forecasterId: forecasterId
        };
        let requestPath = 'forecast.startForecaster';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - startForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        // Extract required parameters with specific type assertion.
        // 'forecasterId' is expected to always be present in the route path.
        let {
          forecasterId
        } = request.params;
        // Extract optional parameters separately.
        // 'dataSourceId' might be missing from the route path (hence '?').
        // Provide a default value ('') if it's not present using destructuring default assignment.
        const {
          dataSourceId = ''
        } = request.params;
        const requestPath = 'forecast.stopForecaster';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          forecasterId: forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - stopForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasterProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('forecast.forecasterProfile', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Forecast - forecasterProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('forecast.searchForecaster', {
          body: requestBody
        });
        const totalForecasters = (0, _lodash.get)(response, 'hits.total.value', 0);
        const forecasters = (0, _lodash.get)(response, 'hits.hits', []).map(forecaster => ({
          ...(0, _forecastHelpers.convertForecastKeysToCamelCase)(forecaster._source),
          id: forecaster._id,
          seqNo: forecaster._seq_no,
          primaryTerm: forecaster._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalForecasters,
              forecasters
            }
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to search forecasters', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasters: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasters", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.searchForecaster', {
          body: {
            size: _constants.MAX_FORECASTER
          }
        });
        const totalForecasters = (0, _lodash.get)(response, 'hits.total.value', 0);

        //Get all forecasters from search forecaster API
        const allForecasters = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, forecasterResponse) => ({
          ...acc,
          [forecasterResponse._id]: {
            id: forecasterResponse._id,
            primaryTerm: forecasterResponse._primary_term,
            seqNo: forecasterResponse._seq_no,
            ...(0, _forecastHelpers.convertStaticFieldsToCamelCase)(forecasterResponse._source)
          }
        }), {});

        // Fetch the latest realtime and runOnce tasks for all forecasters
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no forecaster jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let runOnceTasksResponse = {};
        try {
          realtimeTasksResponse = await callWithRequest('forecast.searchTasks', {
            body: (0, _forecastHelpers.getLatestForecasterTasksQuery)(true)
          });
          runOnceTasksResponse = await callWithRequest('forecast.searchTasks', {
            body: (0, _forecastHelpers.getLatestForecasterTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _forecastHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.forecasters.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const runOnceTasks = (0, _lodash.get)(runOnceTasksResponse, 'aggregations.forecasters.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              runOnceTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});

        // Get real-time and runOnce task info by looping through each forecaster & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id
        const forecastersArray = Object.values(allForecasters);
        forecastersArray.forEach(forecaster => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[forecaster.id], 'realtimeTask._source');
          const runOnceTask = (0, _lodash.get)(runOnceTasks[forecaster.id], 'runOnceTask._source');
          forecaster.curState = (0, _forecastHelpers.combineTaskState)(realtimeTask, runOnceTask);
          forecaster.realTimeLastUpdateTime = (0, _lodash.get)(realtimeTask, 'last_update_time');
          forecaster.runOnceLastUpdateTime = (0, _lodash.get)(runOnceTask, 'last_update_time');
          forecaster.stateError = (0, _lodash.get)(realtimeTask, 'error') || (0, _lodash.get)(runOnceTask, 'error');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalForecasters: totalForecasters,
              forecasterList: forecastersArray
            }
          }
        });
      } catch (err) {
        console.log('Forecaster - Unable to search forecasters', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasterList: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecastResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isRunOnce,
        resultIndex
      } = request.params;
      const {
        dataSourceId = ''
      } = request.params;
      if (!resultIndex) {
        // Not strictly querying custom ⇒ default to ''
        resultIndex = '';
      } else if (!resultIndex.startsWith(_constants.CUSTOM_FORECAST_RESULT_INDEX_PREFIX)) {
        // If resultIndex is given but not valid, revert to default
        resultIndex = '';
      }
      isRunOnce = JSON.parse(isRunOnce);

      // Search by task id if runOnce, or by forecaster id if realtime
      const searchTerm = isRunOnce ? {
        task_id: id
      } : {
        forecaster_id: id
      };
      try {
        const {
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.FORECASTER_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          entityList = '',
          dawnEpoch = 0,
          maxEntities = 0
        } = request.query;

        //Allowed sorting columns
        const sortQueryMap = {
          [_constants.FORECASTER_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.FORECASTER_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.FORECASTER_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.FORECASTER_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }

        //Preparing search request
        const requestBody = {
          sort,
          size,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }]
            }
          }
        };

        // If querying RT results: remove any results that include a task_id, as this indicates
        // a runOnce result from a runOnce task.
        if (!isRunOnce) {
          requestBody.query.bool = {
            ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }
        try {
          // Get current number of filters to determine the index for adding new date range filter
          // This includes the initial term filter and any entity filters that were added
          let filterSize = requestBody.query.bool.filter.length;
          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
          filterSize = requestBody.query.bool.filter.length;

          // Add dawnEpoch filter if it exists
          if (dawnEpoch > 0) {
            (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${_constants.FORECASTER_DOC_FIELDS.EXECUTION_END_TIME}.gte`, dawnEpoch);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }

        // ─────────────────────────────────────────────────────────────
        // If maxEntities > 0, find top N entity_ids.
        // ─────────────────────────────────────────────────────────────
        let restrictedEntityIds = [];
        if (maxEntities > 0) {
          const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
          const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? {} : (0, _forecastHelpers.buildEntityListQuery)(entityListAsObj);

          // Only clone and modify requestBody if entityFilters exists and is not empty/null
          let queryForAggregation;
          if (entityFilters && typeof entityFilters === 'object' && Object.keys(entityFilters).length > 0) {
            // Create a deep clone of the request body
            const clonedRequestBody = JSON.parse(JSON.stringify(requestBody));

            // Add entity filters to the filter array of the cloned request body
            if (!clonedRequestBody.query) {
              clonedRequestBody.query = {
                bool: {
                  filter: []
                }
              };
            } else if (!clonedRequestBody.query.bool) {
              clonedRequestBody.query.bool = {
                filter: []
              };
            } else if (!clonedRequestBody.query.bool.filter) {
              clonedRequestBody.query.bool.filter = [];
            }

            // Add the entity filter object to the filter array
            clonedRequestBody.query.bool.filter.push(entityFilters);
            queryForAggregation = clonedRequestBody.query;
          } else {
            // Use the original requestBody if no entity filters to add
            queryForAggregation = requestBody.query;
          }

          // Example aggregatorRequestBody:
          // {
          //   "size": 0,
          //   "query": {
          //     "bool": {
          //       "filter": [
          //         {"term": {"task_id": "BsLQbZUBxkwQb14j93bF"}},
          //         {"range": {"execution_end_time": {"gte": "0"}}},
          //         {
          //           "bool": {
          //             "should": [
          //               {
          //                 "bool": {
          //                   "must": [
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "service"}}, {"term": {"entity.value": "app_6"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     },
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "host"}}, {"term": {"entity.value": "server_3"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     }
          //                   ]
          //                 }
          //               },
          //               {
          //                 "bool": {
          //                   "must": [
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "service"}}, {"term": {"entity.value": "app_6"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     },
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "host"}}, {"term": {"entity.value": "server_1"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     }
          //                   ]
          //                 }
          //               }
          //             ],
          //             "minimum_should_match": 1
          //           }
          //         }
          //       ]
          //     }
          //   },
          //   "aggs": {
          //     "top_entities": {
          //       "terms": {
          //         "field": "entity_id",
          //         "size": 5,
          //         "order": {"_count": "desc"}
          //       }
          //     }
          //   }
          // }

          // Now use the appropriate query in aggregatorRequestBody
          const aggregatorRequestBody = {
            size: 0,
            query: queryForAggregation,
            aggs: {
              top_entities: {
                terms: {
                  field: 'entity_id',
                  size: maxEntities,
                  order: {
                    _count: 'desc'
                  }
                }
              }
            }
          };

          // We'll call the same or custom search method:
          const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
          const aggResponse = !resultIndex ? await callWithRequest('forecast.searchResults', {
            body: aggregatorRequestBody
          }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
            resultIndex: resultIndex,
            body: aggregatorRequestBody
          });

          // Extract top entity_ids
          const topEntityBuckets = (0, _lodash.get)(aggResponse, 'aggregations.top_entities.buckets', []);
          restrictedEntityIds = topEntityBuckets.map(b => b.key);

          // If no entities matched, return empty
          if (!restrictedEntityIds.length) {
            return opensearchDashboardsResponse.ok({
              body: {
                ok: true,
                response: {
                  totalResults: 0,
                  results: []
                }
              }
            });
          }
        }

        // ─────────────────────────────────────────────────────────────
        // Add a terms filter to restrict final hits if we have top entities
        // ─────────────────────────────────────────────────────────────
        if (restrictedEntityIds.length > 0) {
          requestBody.query.bool.filter.push({
            terms: {
              entity_id: restrictedEntityIds
            }
          });
        }
        let requestParams = {
          resultIndex: resultIndex
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);

        // Add pagination with search_after
        let allResults = [];
        let lastSort = null;
        let hasMoreResults = true;
        let totalHits = 0;

        // Create a copy of your existing requestBody to use in the pagination loop
        const paginatedRequestBody = {
          ...requestBody,
          "track_total_hits": true // Add this to ensure accurate total count for large result sets
        };

        // Add sort if not already present
        if (!paginatedRequestBody.sort) {
          paginatedRequestBody.sort = [{
            [sortField]: sortDirection.toLowerCase()
          }, {
            "_id": "asc"
          } // Secondary sort for tiebreaker
          ];
        }

        // Execute paginated search
        while (hasMoreResults) {
          // Add search_after for subsequent pages
          if (lastSort) {
            paginatedRequestBody.search_after = lastSort;
          }

          // Your existing API call, but with our paginated request body
          const response = !resultIndex ? await callWithRequest('forecast.searchResults', {
            body: paginatedRequestBody
          }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
            ...requestParams,
            body: paginatedRequestBody
          });
          const hits = (0, _lodash.get)(response, 'hits.hits', []);

          // Track total hits from first page
          if (!lastSort) {
            totalHits = (0, _lodash.get)(response, 'hits.total.value', 0);
          }
          if (hits.length === 0 || hits.length < size) {
            hasMoreResults = false;
          }
          if (hits.length > 0) {
            // Save sort values from last hit for next iteration
            lastSort = hits[hits.length - 1].sort;

            // Collect results
            allResults.push(...hits);
          }
        }
        const groupedResults = new Map();
        allResults.forEach(result => {
          const source = result._source;
          const key = `${source.forecaster_id}|${source.entity_id || 'default'}|${source.data_end_time}`;
          if (!groupedResults.has(key)) {
            groupedResults.set(key, {
              featureData: null,
              forecasts: []
            });
          }
          if (source.feature_data) {
            groupedResults.get(key).featureData = result;
          } else {
            groupedResults.get(key).forecasts.push(result);
          }
        });
        const forecastResult = [];

        // Process each group
        groupedResults.forEach(({
          featureData,
          forecasts
        }) => {
          if (!featureData) return; // Skip if no feature data found

          // Check if any forecast has horizon_index
          const hasHorizonIndex = forecasts.some(forecast => forecast._source.horizon_index != null);
          if (hasHorizonIndex) {
            // Sort forecasts by horizon_index and combine into arrays
            const sortedForecasts = (0, _lodash.orderBy)(forecasts, ['_source.horizon_index'], ['asc']);
            const forecastValues = [];
            const forecastLowerBounds = [];
            const forecastUpperBounds = [];
            const forecastStartTimes = [];
            const forecastEndTimes = [];
            sortedForecasts.forEach(forecast => {
              const source = forecast._source;
              forecastValues.push(source.forecast_value != null && source.forecast_value !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_value)) : 0);
              forecastLowerBounds.push(source.forecast_lower_bound != null && source.forecast_lower_bound !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_lower_bound)) : 0);
              forecastUpperBounds.push(source.forecast_upper_bound != null && source.forecast_upper_bound !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_upper_bound)) : 0);
              forecastStartTimes.push(source.forecast_data_start_time);
              forecastEndTimes.push(source.forecast_data_end_time);
            });
            forecastResult.push({
              startTime: featureData._source.data_start_time,
              endTime: featureData._source.data_end_time,
              plotTime: featureData._source.data_end_time,
              forecastValue: forecastValues,
              forecastLowerBound: forecastLowerBounds,
              forecastUpperBound: forecastUpperBounds,
              forecastStartTime: forecastStartTimes,
              forecastEndTime: forecastEndTimes,
              ...(featureData._source.entity != null ? {
                entity: featureData._source.entity,
                entityId: featureData._source.entity_id
              } : {}),
              features: this.getFeatureData(featureData)
            });
          } else {
            // Direct push for single forecasts without horizon_index
            forecastResult.push({
              startTime: featureData._source.data_start_time,
              endTime: featureData._source.data_end_time,
              plotTime: featureData._source.data_end_time,
              forecastValue: [],
              forecastLowerBound: [],
              forecastUpperBound: [],
              forecastStartTime: [],
              forecastEndTime: [],
              ...(featureData._source.entity != null ? {
                entity: featureData._source.entity,
                entityId: featureData._source.entity_id
              } : {}),
              features: this.getFeatureData(featureData)
            });
          }
        });

        // Sort final results by plotTime
        const sortedForecastResult = (0, _lodash.orderBy)(forecastResult, ['plotTime'], ['asc']);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalResults: totalHits,
              results: sortedForecastResult
            }
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getTopForecastResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          forecasterId,
          isRunOnce
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isRunOnce = JSON.parse(isRunOnce);
        const requestPath = 'forecast.topForecastResults';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);

        // Define proper types for OpenSearch query structures

        // Extract the query parameters from the request body with defaults
        const {
          split_by = '',
          filter_by = '',
          build_in_query = '',
          forecast_from = 0,
          threshold = 0,
          relation_to_threshold = '',
          filter_query = {},
          subaggregations = []
        } = request.body || {};

        // Build query object with appropriate parameters
        const queryBody = {};

        // Add split_by if present
        if (split_by) {
          queryBody.split_by = split_by;
        }

        // Add filter_by and related parameters
        if (filter_by) {
          queryBody.filter_by = filter_by;
          if (filter_by === 'BUILD_IN_QUERY' && build_in_query) {
            queryBody.build_in_query = build_in_query;

            // Add threshold parameters if build_in_query is DISTANCE_TO_THRESHOLD_VALUE
            if (build_in_query === 'DISTANCE_TO_THRESHOLD_VALUE') {
              queryBody.threshold = threshold;
              queryBody.relation_to_threshold = relation_to_threshold;
            }
          } else if (filter_by === 'CUSTOM_QUERY') {
            // Add custom query parameters - check if the objects are not empty
            if (Object.keys(filter_query).length > 0) {
              queryBody.filter_query = filter_query;
            }
            if (subaggregations.length > 0) {
              queryBody.subaggregations = subaggregations;
            }
          }
        }

        // Add forecast_from timestamp if present
        if (forecast_from) {
          queryBody.forecast_from = forecast_from;
        }

        // Add run_once to body if isRunOnce is true
        const requestBody = {
          ...queryBody,
          ...(isRunOnce && {
            run_once: true
          })
        };
        const response = await callWithRequest(requestPath, {
          forecasterId: forecasterId,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - getTopForecastResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "matchForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterName
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.matchForecaster', {
          forecasterName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - matchForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasterCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.forecasterCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - getForecasterCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};
      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name
        };
      });
      return featureResult;
    });
    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_FORECAST_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search forecast result from default index.
          resultIndex = '';
        }
        let requestParams = {
          resultIndex: resultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('forecast.searchResults', {
          body: requestBody
        }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to search forecast result', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasters: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = ForecastService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfbG9kYXNoIiwicmVxdWlyZSIsIl9jb25zdGFudHMiLCJfaGVscGVycyIsIl9mb3JlY2FzdEhlbHBlcnMiLCJfZGVmaW5lUHJvcGVydHkiLCJlIiwiciIsInQiLCJfdG9Qcm9wZXJ0eUtleSIsIk9iamVjdCIsImRlZmluZVByb3BlcnR5IiwidmFsdWUiLCJlbnVtZXJhYmxlIiwiY29uZmlndXJhYmxlIiwid3JpdGFibGUiLCJpIiwiX3RvUHJpbWl0aXZlIiwiU3ltYm9sIiwidG9QcmltaXRpdmUiLCJjYWxsIiwiVHlwZUVycm9yIiwiU3RyaW5nIiwiTnVtYmVyIiwicmVnaXN0ZXJGb3JlY2FzdFJvdXRlcyIsImFwaVJvdXRlciIsImZvcmVjYXN0U2VydmljZSIsInBvc3QiLCJwdXRGb3JlY2FzdGVyIiwicHV0Iiwic2VhcmNoRm9yZWNhc3RlciIsInNlYXJjaFJlc3VsdHMiLCJnZXQiLCJnZXRGb3JlY2FzdGVycyIsInJ1bk9uY2VGb3JlY2FzdGVyIiwiZ2V0Rm9yZWNhc3RSZXN1bHRzIiwiZGVsZXRlIiwiZGVsZXRlRm9yZWNhc3RlciIsInN0YXJ0Rm9yZWNhc3RlciIsInN0b3BGb3JlY2FzdGVyIiwiZ2V0Rm9yZWNhc3RlclByb2ZpbGUiLCJnZXRGb3JlY2FzdGVyIiwibWF0Y2hGb3JlY2FzdGVyIiwiZ2V0Rm9yZWNhc3RlckNvdW50IiwiZ2V0VG9wRm9yZWNhc3RSZXN1bHRzIiwidmFsaWRhdGVGb3JlY2FzdGVyIiwic3VnZ2VzdEZvcmVjYXN0ZXIiLCJGb3JlY2FzdFNlcnZpY2UiLCJjb25zdHJ1Y3RvciIsImNsaWVudCIsImRhdGFTb3VyY2VFbmFibGVkIiwiY29udGV4dCIsInJlcXVlc3QiLCJvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlIiwiZm9yZWNhc3RlcklkIiwicGFyYW1zIiwiZGF0YVNvdXJjZUlkIiwiY2FsbFdpdGhSZXF1ZXN0IiwiZ2V0Q2xpZW50QmFzZWRPbkRhdGFTb3VyY2UiLCJyZXNwb25zZSIsIm9rIiwiYm9keSIsImVyciIsImNvbnNvbGUiLCJsb2ciLCJlcnJvciIsImdldEVycm9yTWVzc2FnZSIsImlmU2VxTm8iLCJzZXFObyIsImlmUHJpbWFyeVRlcm0iLCJwcmltYXJ5VGVybSIsInJlcXVlc3RCb2R5IiwiSlNPTiIsInN0cmluZ2lmeSIsImNvbnZlcnRGb3JlY2FzdEtleXNUb1NuYWtlQ2FzZSIsImlzTnVtYmVyIiwicmVzcCIsImZvcmVjYXN0ZXIiLCJpZCIsIl9pZCIsIl9wcmltYXJ5X3Rlcm0iLCJfc2VxX25vIiwiY29udmVydEZvcmVjYXN0S2V5c1RvQ2FtZWxDYXNlIiwidmFsaWRhdGlvblR5cGUiLCJzdWdnZXN0VHlwZSIsImZvcmVjYXN0ZXJSZXNwb25zZSIsInN0YXRpY0ZpZWxkcyIsImNvbnZlcnRTdGF0aWNGaWVsZHNUb0NhbWVsQ2FzZSIsInRhc2tBbmRKb2JGaWVsZHMiLCJjb252ZXJ0VGFza0FuZEpvYkZpZWxkc1RvQ2FtZWxDYXNlIiwicmVhbHRpbWVfdGFzayIsInJ1bl9vbmNlX3Rhc2siLCJmb3JlY2FzdGVyX2pvYiIsImZpbmFsUmVzcG9uc2UiLCJzdGF0dXNDb2RlIiwiX3JlcXVlc3QkYm9keSIsIl9yZXF1ZXN0JGJvZHkyIiwic3RhcnRUaW1lIiwiZW5kVGltZSIsInJlcXVlc3RQYXJhbXMiLCJyZXF1ZXN0UGF0aCIsImFzU2NvcGVkIiwiY2FsbEFzQ3VycmVudFVzZXIiLCJ0b3RhbEZvcmVjYXN0ZXJzIiwiZm9yZWNhc3RlcnMiLCJtYXAiLCJfc291cmNlIiwiaXNJbmRleE5vdEZvdW5kRXJyb3IiLCJzaXplIiwiTUFYX0ZPUkVDQVNURVIiLCJhbGxGb3JlY2FzdGVycyIsInJlZHVjZSIsImFjYyIsInJlYWx0aW1lVGFza3NSZXNwb25zZSIsInJ1bk9uY2VUYXNrc1Jlc3BvbnNlIiwiZ2V0TGF0ZXN0Rm9yZWNhc3RlclRhc2tzUXVlcnkiLCJyZWFsdGltZVRhc2tzIiwiYnVja2V0Iiwia2V5IiwicmVhbHRpbWVUYXNrIiwidW5kZWZpbmVkIiwicnVuT25jZVRhc2tzIiwicnVuT25jZVRhc2siLCJmb3JlY2FzdGVyc0FycmF5IiwidmFsdWVzIiwiZm9yRWFjaCIsImN1clN0YXRlIiwiY29tYmluZVRhc2tTdGF0ZSIsInJlYWxUaW1lTGFzdFVwZGF0ZVRpbWUiLCJydW5PbmNlTGFzdFVwZGF0ZVRpbWUiLCJzdGF0ZUVycm9yIiwiZm9yZWNhc3Rlckxpc3QiLCJpc1J1bk9uY2UiLCJyZXN1bHRJbmRleCIsInN0YXJ0c1dpdGgiLCJDVVNUT01fRk9SRUNBU1RfUkVTVUxUX0lOREVYX1BSRUZJWCIsInBhcnNlIiwic2VhcmNoVGVybSIsInRhc2tfaWQiLCJmb3JlY2FzdGVyX2lkIiwic29ydERpcmVjdGlvbiIsIlNPUlRfRElSRUNUSU9OIiwiREVTQyIsInNvcnRGaWVsZCIsIkZPUkVDQVNURVJfRE9DX0ZJRUxEUyIsIkRBVEFfU1RBUlRfVElNRSIsImZpZWxkTmFtZSIsImVudGl0eUxpc3QiLCJkYXduRXBvY2giLCJtYXhFbnRpdGllcyIsInF1ZXJ5Iiwic29ydFF1ZXJ5TWFwIiwiREFUQV9FTkRfVElNRSIsInNvcnQiLCJzb3J0UXVlcnkiLCJib29sIiwiZmlsdGVyIiwidGVybSIsIm11c3Rfbm90IiwiZXhpc3RzIiwiZmllbGQiLCJmaWx0ZXJTaXplIiwibGVuZ3RoIiwic2V0IiwiRVhFQ1VUSU9OX0VORF9USU1FIiwicmVzdHJpY3RlZEVudGl0eUlkcyIsImVudGl0eUxpc3RBc09iaiIsImVudGl0eUZpbHRlcnMiLCJpc0VtcHR5IiwiYnVpbGRFbnRpdHlMaXN0UXVlcnkiLCJxdWVyeUZvckFnZ3JlZ2F0aW9uIiwia2V5cyIsImNsb25lZFJlcXVlc3RCb2R5IiwicHVzaCIsImFnZ3JlZ2F0b3JSZXF1ZXN0Qm9keSIsImFnZ3MiLCJ0b3BfZW50aXRpZXMiLCJ0ZXJtcyIsIm9yZGVyIiwiX2NvdW50IiwiYWdnUmVzcG9uc2UiLCJ0b3BFbnRpdHlCdWNrZXRzIiwiYiIsInRvdGFsUmVzdWx0cyIsInJlc3VsdHMiLCJlbnRpdHlfaWQiLCJhbGxSZXN1bHRzIiwibGFzdFNvcnQiLCJoYXNNb3JlUmVzdWx0cyIsInRvdGFsSGl0cyIsInBhZ2luYXRlZFJlcXVlc3RCb2R5IiwidG9Mb3dlckNhc2UiLCJzZWFyY2hfYWZ0ZXIiLCJoaXRzIiwiZ3JvdXBlZFJlc3VsdHMiLCJNYXAiLCJyZXN1bHQiLCJzb3VyY2UiLCJkYXRhX2VuZF90aW1lIiwiaGFzIiwiZmVhdHVyZURhdGEiLCJmb3JlY2FzdHMiLCJmZWF0dXJlX2RhdGEiLCJmb3JlY2FzdFJlc3VsdCIsImhhc0hvcml6b25JbmRleCIsInNvbWUiLCJmb3JlY2FzdCIsImhvcml6b25faW5kZXgiLCJzb3J0ZWRGb3JlY2FzdHMiLCJvcmRlckJ5IiwiZm9yZWNhc3RWYWx1ZXMiLCJmb3JlY2FzdExvd2VyQm91bmRzIiwiZm9yZWNhc3RVcHBlckJvdW5kcyIsImZvcmVjYXN0U3RhcnRUaW1lcyIsImZvcmVjYXN0RW5kVGltZXMiLCJmb3JlY2FzdF92YWx1ZSIsInRvRml4ZWROdW1iZXJGb3JGb3JlY2FzdCIsInBhcnNlRmxvYXQiLCJmb3JlY2FzdF9sb3dlcl9ib3VuZCIsImZvcmVjYXN0X3VwcGVyX2JvdW5kIiwiZm9yZWNhc3RfZGF0YV9zdGFydF90aW1lIiwiZm9yZWNhc3RfZGF0YV9lbmRfdGltZSIsImRhdGFfc3RhcnRfdGltZSIsInBsb3RUaW1lIiwiZm9yZWNhc3RWYWx1ZSIsImZvcmVjYXN0TG93ZXJCb3VuZCIsImZvcmVjYXN0VXBwZXJCb3VuZCIsImZvcmVjYXN0U3RhcnRUaW1lIiwiZm9yZWNhc3RFbmRUaW1lIiwiZW50aXR5IiwiZW50aXR5SWQiLCJmZWF0dXJlcyIsImdldEZlYXR1cmVEYXRhIiwic29ydGVkRm9yZWNhc3RSZXN1bHQiLCJzcGxpdF9ieSIsImZpbHRlcl9ieSIsImJ1aWxkX2luX3F1ZXJ5IiwiZm9yZWNhc3RfZnJvbSIsInRocmVzaG9sZCIsInJlbGF0aW9uX3RvX3RocmVzaG9sZCIsImZpbHRlcl9xdWVyeSIsInN1YmFnZ3JlZ2F0aW9ucyIsInF1ZXJ5Qm9keSIsInJ1bl9vbmNlIiwiZm9yZWNhc3Rlck5hbWUiLCJyYXdSZXN1bHQiLCJmZWF0dXJlUmVzdWx0IiwiZmVhdHVyZV9pZCIsImRhdGEiLCJuYW1lIiwiZmVhdHVyZV9uYW1lIiwiZXhwb3J0cyIsImRlZmF1bHQiXSwic291cmNlcyI6WyJmb3JlY2FzdC50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqXG4gKiBUaGUgT3BlblNlYXJjaCBDb250cmlidXRvcnMgcmVxdWlyZSBjb250cmlidXRpb25zIG1hZGUgdG9cbiAqIHRoaXMgZmlsZSBiZSBsaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlLTIuMCBsaWNlbnNlIG9yIGFcbiAqIGNvbXBhdGlibGUgb3BlbiBzb3VyY2UgbGljZW5zZS5cbiAqXG4gKiBNb2RpZmljYXRpb25zIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9ycy4gU2VlXG4gKiBHaXRIdWIgaGlzdG9yeSBmb3IgZGV0YWlscy5cbiAqL1xuXG5pbXBvcnQgeyBnZXQsIG9yZGVyQnksIGlzRW1wdHkgfSBmcm9tICdsb2Rhc2gnO1xuaW1wb3J0IHsgU2VhcmNoUmVzcG9uc2UgfSBmcm9tICcuLi9tb2RlbHMvaW50ZXJmYWNlcyc7XG5pbXBvcnQge1xuICBGb3JlY2FzdFJlc3VsdCxcbiAgRm9yZWNhc3RlcixcbiAgRmVhdHVyZVJlc3VsdCxcbiAgR2V0Rm9yZWNhc3RlcnNRdWVyeVBhcmFtcyxcbn0gZnJvbSAnLi4vbW9kZWxzL3R5cGVzJztcbmltcG9ydCB7IFJvdXRlciB9IGZyb20gJy4uL3JvdXRlcic7XG5pbXBvcnQge1xuICBTT1JUX0RJUkVDVElPTixcbiAgQ1VTVE9NX0ZPUkVDQVNUX1JFU1VMVF9JTkRFWF9QUkVGSVgsXG4gIE1BWF9GT1JFQ0FTVEVSLFxuICBGT1JFQ0FTVEVSX0RPQ19GSUVMRFMsXG59IGZyb20gJy4uL3V0aWxzL2NvbnN0YW50cyc7XG5pbXBvcnQge1xuICBnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZSxcbiAgdG9GaXhlZE51bWJlckZvckZvcmVjYXN0LFxufSBmcm9tICcuLi91dGlscy9oZWxwZXJzJztcbmltcG9ydCB7XG4gIGNvbnZlcnRGb3JlY2FzdEtleXNUb0NhbWVsQ2FzZSxcbiAgY29udmVydEZvcmVjYXN0S2V5c1RvU25ha2VDYXNlLFxuICBpc0luZGV4Tm90Rm91bmRFcnJvcixcbiAgZ2V0RXJyb3JNZXNzYWdlLFxuICBnZXRMYXRlc3RGb3JlY2FzdGVyVGFza3NRdWVyeSxcbiAgYnVpbGRFbnRpdHlMaXN0UXVlcnksXG4gIGNvbnZlcnRTdGF0aWNGaWVsZHNUb0NhbWVsQ2FzZSxcbiAgY29udmVydFRhc2tBbmRKb2JGaWVsZHNUb0NhbWVsQ2FzZSxcbiAgY29tYmluZVRhc2tTdGF0ZSxcbn0gZnJvbSAnLi91dGlscy9mb3JlY2FzdEhlbHBlcnMnO1xuaW1wb3J0IHsgaXNOdW1iZXIsIHNldCB9IGZyb20gJ2xvZGFzaCc7XG5pbXBvcnQge1xuICBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gIE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnksXG4gIElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLFxufSBmcm9tICcuLi8uLi8uLi8uLi9zcmMvY29yZS9zZXJ2ZXInO1xuXG50eXBlIFB1dEZvcmVjYXN0ZXJQYXJhbXMgPSB7XG4gIGZvcmVjYXN0ZXJJZDogc3RyaW5nO1xuICBpZlNlcU5vPzogc3RyaW5nO1xuICBpZlByaW1hcnlUZXJtPzogc3RyaW5nO1xuICBib2R5OiBzdHJpbmc7XG59O1xuXG5cbmV4cG9ydCBmdW5jdGlvbiByZWdpc3RlckZvcmVjYXN0Um91dGVzKGFwaVJvdXRlcjogUm91dGVyLCBmb3JlY2FzdFNlcnZpY2U6IEZvcmVjYXN0U2VydmljZSkge1xuICAvLyBjcmVhdGUgZm9yZWNhc3RlclxuICBhcGlSb3V0ZXIucG9zdCgnL2ZvcmVjYXN0ZXJzJywgZm9yZWNhc3RTZXJ2aWNlLnB1dEZvcmVjYXN0ZXIpO1xuICBhcGlSb3V0ZXIucG9zdCgnL2ZvcmVjYXN0ZXJzL3tkYXRhU291cmNlSWR9JywgZm9yZWNhc3RTZXJ2aWNlLnB1dEZvcmVjYXN0ZXIpO1xuXG4gIC8vIHB1dCBmb3JlY2FzdGVyXG4gIGFwaVJvdXRlci5wdXQoJy9mb3JlY2FzdGVycy97Zm9yZWNhc3RlcklkfScsIGZvcmVjYXN0U2VydmljZS5wdXRGb3JlY2FzdGVyKTtcbiAgYXBpUm91dGVyLnB1dChcbiAgICAnL2ZvcmVjYXN0ZXJzL3tmb3JlY2FzdGVySWR9L3tkYXRhU291cmNlSWR9JyxcbiAgICBmb3JlY2FzdFNlcnZpY2UucHV0Rm9yZWNhc3RlclxuICApO1xuXG4gIC8vIEZJWE1FOiByb3V0ZXMgbm90IHVzZWQgaW4gdGhlIFVJLCB0aGVyZWZvcmUgbm8gZGF0YSBzb3VyY2UgaWRcbiAgYXBpUm91dGVyLnBvc3QoJy9mb3JlY2FzdGVycy9fc2VhcmNoJywgZm9yZWNhc3RTZXJ2aWNlLnNlYXJjaEZvcmVjYXN0ZXIpO1xuXG4gIC8qKlxuICAgKiBTZWFyY2ggZm9yZWNhc3QgcmVzdWx0cyByb3V0ZXNcbiAgICogXG4gICAqIFdlIHVzZSAnYnktc291cmNlJyBhbmQgJ2J5LWluZGV4JyBwYXRoIHNlZ21lbnRzIHRvIGF2b2lkIHJvdXRlIGNvbmZsaWN0cyBiZXR3ZWVuXG4gICAqIHBhdGhzIHdpdGggZGlmZmVyZW50IHBhcmFtZXRlciB0eXBlcy4gV2l0aG91dCB0aGVzZSBzZWdtZW50cywgcm91dGVzIGxpa2U6XG4gICAqICAgL2ZvcmVjYXN0ZXJzL3Jlc3VsdHMvX3NlYXJjaC97cmVzdWx0SW5kZXh9XG4gICAqICAgL2ZvcmVjYXN0ZXJzL3Jlc3VsdHMvX3NlYXJjaC97ZGF0YVNvdXJjZUlkfVxuICAgKiB3b3VsZCBjb25mbGljdCBiZWNhdXNlIE9wZW5TZWFyY2ggY2FuJ3QgZGlzdGluZ3Vpc2ggYmV0d2VlbiBwYXJhbWV0ZXIgdHlwZXMgaW4gdGhlIHNhbWUgcG9zaXRpb24uXG4gICAqIFxuICAgKiBDdXJyZW50IHJvdXRlIHN0cnVjdHVyZTpcbiAgICogMS4gU2VhcmNoIGJ5IHNvdXJjZSAobm8gcGFyYW1zKSAgICAgOiAvYnktc291cmNlL19zZWFyY2hcbiAgICogMi4gU2VhcmNoIGJ5IHNvdXJjZSB3aXRoIElEICAgICAgICAgOiAvYnktc291cmNlL3tkYXRhU291cmNlSWR9L19zZWFyY2hcbiAgICogMy4gU2VhcmNoIGJ5IGluZGV4IHBhdHRlcm4gICAgICAgICAgOiAvYnktaW5kZXgve3Jlc3VsdEluZGV4fS9fc2VhcmNoXG4gICAqIDQuIFNlYXJjaCBieSBib3RoIGluZGV4IGFuZCBzb3VyY2UgIDogL2J5LWluZGV4L3tyZXN1bHRJbmRleH0vYnktc291cmNlL3tkYXRhU291cmNlSWR9L19zZWFyY2hcbiAgICovXG5cbiAgLy8gU2VhcmNoIHdpdGggbm8gcGFyYW1ldGVyc1xuICBhcGlSb3V0ZXIucG9zdCgnL2ZvcmVjYXN0ZXJzL3Jlc3VsdHMvYnktc291cmNlL19zZWFyY2gnLCBmb3JlY2FzdFNlcnZpY2Uuc2VhcmNoUmVzdWx0cyk7XG5cbiAgLy8gU2VhcmNoIGJ5IGRhdGEgc291cmNlIElEXG4gIGFwaVJvdXRlci5wb3N0KFxuICAgICcvZm9yZWNhc3RlcnMvcmVzdWx0cy9ieS1zb3VyY2Uve2RhdGFTb3VyY2VJZH0vX3NlYXJjaCcsXG4gICAgZm9yZWNhc3RTZXJ2aWNlLnNlYXJjaFJlc3VsdHNcbiAgKTtcblxuICAvLyBTZWFyY2ggYnkgcmVzdWx0IGluZGV4IHBhdHRlcm5cbiAgYXBpUm91dGVyLnBvc3QoXG4gICAgJy9mb3JlY2FzdGVycy9yZXN1bHRzL2J5LWluZGV4L3tyZXN1bHRJbmRleH0vX3NlYXJjaCcsXG4gICAgZm9yZWNhc3RTZXJ2aWNlLnNlYXJjaFJlc3VsdHNcbiAgKTtcblxuICAvLyBTZWFyY2ggYnkgYm90aCByZXN1bHQgaW5kZXggYW5kIGRhdGEgc291cmNlIElEXG4gIGFwaVJvdXRlci5wb3N0KFxuICAgICcvZm9yZWNhc3RlcnMvcmVzdWx0cy9ieS1pbmRleC97cmVzdWx0SW5kZXh9L2J5LXNvdXJjZS97ZGF0YVNvdXJjZUlkfS9fc2VhcmNoJyxcbiAgICBmb3JlY2FzdFNlcnZpY2Uuc2VhcmNoUmVzdWx0c1xuICApO1xuXG4gIC8vIGxpc3QgZm9yZWNhc3RlcnNcbiAgYXBpUm91dGVyLmdldCgnL2ZvcmVjYXN0ZXJzL19saXN0JywgZm9yZWNhc3RTZXJ2aWNlLmdldEZvcmVjYXN0ZXJzKTtcbiAgYXBpUm91dGVyLmdldCgnL2ZvcmVjYXN0ZXJzL19saXN0L3tkYXRhU291cmNlSWR9JywgZm9yZWNhc3RTZXJ2aWNlLmdldEZvcmVjYXN0ZXJzKTtcblxuICAvLyBydW4gb25jZSBmb3JlY2FzdGVyXG4gIGFwaVJvdXRlci5wb3N0KCcvZm9yZWNhc3RlcnMve2ZvcmVjYXN0ZXJJZH0vX3J1bl9vbmNlJywgZm9yZWNhc3RTZXJ2aWNlLnJ1bk9uY2VGb3JlY2FzdGVyKTtcbiAgYXBpUm91dGVyLnBvc3QoJy9mb3JlY2FzdGVycy97Zm9yZWNhc3RlcklkfS9fcnVuX29uY2Uve2RhdGFTb3VyY2VJZH0nLCBmb3JlY2FzdFNlcnZpY2UucnVuT25jZUZvcmVjYXN0ZXIpO1xuXG4gIC8vIGdldCBmb3JlY2FzdGVyIGZvcmVjYXN0IHJlc3VsdHNcbiAgYXBpUm91dGVyLmdldChcbiAgICAnL2ZvcmVjYXN0ZXJzL3tpZH0vcmVzdWx0cy97aXNSdW5PbmNlfS97cmVzdWx0SW5kZXh9JyxcbiAgICBmb3JlY2FzdFNlcnZpY2UuZ2V0Rm9yZWNhc3RSZXN1bHRzXG4gICk7XG4gIGFwaVJvdXRlci5nZXQoXG4gICAgJy9mb3JlY2FzdGVycy97aWR9L3Jlc3VsdHMve2lzUnVuT25jZX0ve3Jlc3VsdEluZGV4fS97ZGF0YVNvdXJjZUlkfScsXG4gICAgZm9yZWNhc3RTZXJ2aWNlLmdldEZvcmVjYXN0UmVzdWx0c1xuICApO1xuXG4gIC8vIGRlbGV0ZSBmb3JlY2FzdGVyXG4gIGFwaVJvdXRlci5kZWxldGUoJy9mb3JlY2FzdGVycy97Zm9yZWNhc3RlcklkfScsIGZvcmVjYXN0U2VydmljZS5kZWxldGVGb3JlY2FzdGVyKTtcbiAgYXBpUm91dGVyLmRlbGV0ZShcbiAgICAnL2ZvcmVjYXN0ZXJzL3tmb3JlY2FzdGVySWR9L3tkYXRhU291cmNlSWR9JyxcbiAgICBmb3JlY2FzdFNlcnZpY2UuZGVsZXRlRm9yZWNhc3RlclxuICApO1xuXG4gIC8vIHN0YXJ0IGZvcmVjYXN0ZXJcbiAgYXBpUm91dGVyLnBvc3QoJy9mb3JlY2FzdGVycy97Zm9yZWNhc3RlcklkfS9zdGFydCcsIGZvcmVjYXN0U2VydmljZS5zdGFydEZvcmVjYXN0ZXIpO1xuICBhcGlSb3V0ZXIucG9zdChcbiAgICAnL2ZvcmVjYXN0ZXJzL3tmb3JlY2FzdGVySWR9L3N0YXJ0L3tkYXRhU291cmNlSWR9JyxcbiAgICBmb3JlY2FzdFNlcnZpY2Uuc3RhcnRGb3JlY2FzdGVyXG4gICk7XG5cbiAgLy8gc3RvcCBmb3JlY2FzdGVyXG4gIGFwaVJvdXRlci5wb3N0KFxuICAgICcvZm9yZWNhc3RlcnMve2ZvcmVjYXN0ZXJJZH0vc3RvcCcsXG4gICAgZm9yZWNhc3RTZXJ2aWNlLnN0b3BGb3JlY2FzdGVyXG4gICk7XG4gIGFwaVJvdXRlci5wb3N0KFxuICAgICcvZm9yZWNhc3RlcnMve2ZvcmVjYXN0ZXJJZH0vc3RvcC97ZGF0YVNvdXJjZUlkfScsXG4gICAgZm9yZWNhc3RTZXJ2aWNlLnN0b3BGb3JlY2FzdGVyXG4gICk7XG5cbiAgYXBpUm91dGVyLmdldChcbiAgICAnL2ZvcmVjYXN0ZXJzL3tmb3JlY2FzdGVySWR9L19wcm9maWxlJyxcbiAgICBmb3JlY2FzdFNlcnZpY2UuZ2V0Rm9yZWNhc3RlclByb2ZpbGVcbiAgKTtcblxuICAvLyBnZXQgZm9yZWNhc3RlclxuICBhcGlSb3V0ZXIuZ2V0KCcvZm9yZWNhc3RlcnMve2ZvcmVjYXN0ZXJJZH0nLCBmb3JlY2FzdFNlcnZpY2UuZ2V0Rm9yZWNhc3Rlcik7XG4gIGFwaVJvdXRlci5nZXQoXG4gICAgJy9mb3JlY2FzdGVycy97Zm9yZWNhc3RlcklkfS97ZGF0YVNvdXJjZUlkfScsXG4gICAgZm9yZWNhc3RTZXJ2aWNlLmdldEZvcmVjYXN0ZXJcbiAgKTtcblxuICAvLyBtYXRjaCBmb3JlY2FzdGVyXG4gIGFwaVJvdXRlci5nZXQoJy9mb3JlY2FzdGVycy97Zm9yZWNhc3Rlck5hbWV9L19tYXRjaCcsIGZvcmVjYXN0U2VydmljZS5tYXRjaEZvcmVjYXN0ZXIpO1xuICBhcGlSb3V0ZXIuZ2V0KCcvZm9yZWNhc3RlcnMve2ZvcmVjYXN0ZXJOYW1lfS9fbWF0Y2gve2RhdGFTb3VyY2VJZH0nLCBmb3JlY2FzdFNlcnZpY2UubWF0Y2hGb3JlY2FzdGVyKTtcblxuICAvLyBnZXQgZm9yZWNhc3RlciBjb3VudFxuICBhcGlSb3V0ZXIuZ2V0KCcvZm9yZWNhc3RlcnMvX2NvdW50JywgZm9yZWNhc3RTZXJ2aWNlLmdldEZvcmVjYXN0ZXJDb3VudCk7XG4gIGFwaVJvdXRlci5nZXQoJy9mb3JlY2FzdGVycy9fY291bnQve2RhdGFTb3VyY2VJZH0nLCBmb3JlY2FzdFNlcnZpY2UuZ2V0Rm9yZWNhc3RlckNvdW50KTtcblxuICAvLyBwb3N0IGdldCB0b3AgZm9yZWNhc3QgcmVzdWx0c1xuICBhcGlSb3V0ZXIucG9zdChcbiAgICAnL2ZvcmVjYXN0ZXJzL3tmb3JlY2FzdGVySWR9L190b3BGb3JlY2FzdHMve2lzUnVuT25jZX0nLFxuICAgIGZvcmVjYXN0U2VydmljZS5nZXRUb3BGb3JlY2FzdFJlc3VsdHNcbiAgKTtcbiAgYXBpUm91dGVyLnBvc3QoXG4gICAgJy9mb3JlY2FzdGVycy97Zm9yZWNhc3RlcklkfS9fdG9wRm9yZWNhc3RzL3tpc1J1bk9uY2V9L3tkYXRhU291cmNlSWR9JyxcbiAgICBmb3JlY2FzdFNlcnZpY2UuZ2V0VG9wRm9yZWNhc3RSZXN1bHRzXG4gICk7XG5cbiAgLy8gdmFsaWRhdGUgZm9yZWNhc3RlclxuICBhcGlSb3V0ZXIucG9zdChcbiAgICAnL2ZvcmVjYXN0ZXJzL192YWxpZGF0ZS97dmFsaWRhdGlvblR5cGV9JyxcbiAgICBmb3JlY2FzdFNlcnZpY2UudmFsaWRhdGVGb3JlY2FzdGVyXG4gICk7XG4gIGFwaVJvdXRlci5wb3N0KFxuICAgICcvZm9yZWNhc3RlcnMvX3ZhbGlkYXRlL3t2YWxpZGF0aW9uVHlwZX0ve2RhdGFTb3VyY2VJZH0nLFxuICAgIGZvcmVjYXN0U2VydmljZS52YWxpZGF0ZUZvcmVjYXN0ZXJcbiAgKTtcblxuICAvLyBzdWdnZXN0IGZvcmVjYXN0ZXJcbiAgYXBpUm91dGVyLnBvc3QoXG4gICAgJy9mb3JlY2FzdGVycy9fc3VnZ2VzdC97c3VnZ2VzdFR5cGV9JyxcbiAgICBmb3JlY2FzdFNlcnZpY2Uuc3VnZ2VzdEZvcmVjYXN0ZXJcbiAgKTtcbiAgYXBpUm91dGVyLnBvc3QoXG4gICAgJy9mb3JlY2FzdGVycy9fc3VnZ2VzdC97c3VnZ2VzdFR5cGV9L3tkYXRhU291cmNlSWR9JyxcbiAgICBmb3JlY2FzdFNlcnZpY2Uuc3VnZ2VzdEZvcmVjYXN0ZXJcbiAgKTtcbn1cblxuZXhwb3J0IGRlZmF1bHQgY2xhc3MgRm9yZWNhc3RTZXJ2aWNlIHtcbiAgcHJpdmF0ZSBjbGllbnQ6IGFueTtcbiAgZGF0YVNvdXJjZUVuYWJsZWQ6IGJvb2xlYW47XG5cbiAgY29uc3RydWN0b3IoY2xpZW50OiBhbnksIGRhdGFTb3VyY2VFbmFibGVkOiBib29sZWFuKSB7XG4gICAgdGhpcy5jbGllbnQgPSBjbGllbnQ7XG4gICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCA9IGRhdGFTb3VyY2VFbmFibGVkO1xuICB9XG5cbiAgZGVsZXRlRm9yZWNhc3RlciA9IGFzeW5jIChcbiAgICBjb250ZXh0OiBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gICAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICAgIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55Pj4gPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGZvcmVjYXN0ZXJJZCB9ID0gcmVxdWVzdC5wYXJhbXMgYXMgeyBmb3JlY2FzdGVySWQ6IHN0cmluZyB9O1xuICAgICAgY29uc3QgeyBkYXRhU291cmNlSWQgPSAnJyB9ID0gcmVxdWVzdC5wYXJhbXMgYXMgeyBkYXRhU291cmNlSWQ/OiBzdHJpbmcgfTtcbiAgICAgIC8vIGRhdGFTb3VyY2VJZCB3aWxsIGJlIFwiXCIgYW5kIGZhbGwgYmFjayB0byB1c2UgdGhlIGV4aXN0aW5nIGNsaWVudCBmb3IgbG9jYWwgY2x1c3RlclxuICAgICAgLy8gT24gdGhlIG90aGVyIGhhbmQsIGluIE1EUyB3b3JsZCwgdGhlIG9wZW4gc2VhcmNoIGxlZ2FjeSBjbGllbnQgKHRoaXMuY2xpZW50KSB3aWxsXG4gICAgICAvLyBiZSB1bmRlZmluZWQgYW5kIGl0IHdpbGwgcGlja3VwIHRoZSBkYXRhIHNvdXJjZSBjbGllbnQgXG4gICAgICBjb25zdCBjYWxsV2l0aFJlcXVlc3QgPSBnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShcbiAgICAgICAgY29udGV4dCxcbiAgICAgICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCxcbiAgICAgICAgcmVxdWVzdCxcbiAgICAgICAgZGF0YVNvdXJjZUlkLFxuICAgICAgICB0aGlzLmNsaWVudFxuICAgICAgKTtcblxuICAgICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoJ2ZvcmVjYXN0LmRlbGV0ZUZvcmVjYXN0ZXInLCB7XG4gICAgICAgIGZvcmVjYXN0ZXJJZCxcbiAgICAgIH0pO1xuXG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZTogcmVzcG9uc2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUubG9nKCdGb3JlY2FzdCAtIGRlbGV0ZUZvcmVjYXN0ZXInLCBlcnIpO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBnZXRFcnJvck1lc3NhZ2UoZXJyKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBydW5PbmNlRm9yZWNhc3RlciA9IGFzeW5jIChcbiAgICBjb250ZXh0OiBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gICAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICAgIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55Pj4gPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGZvcmVjYXN0ZXJJZCA9ICcnLCBkYXRhU291cmNlSWQgPSAnJyB9ID0gcmVxdWVzdC5wYXJhbXMgYXMgeyBmb3JlY2FzdGVySWQ/OiBzdHJpbmcsIGRhdGFTb3VyY2VJZD86IHN0cmluZyB9O1xuXG4gICAgICBjb25zdCBjYWxsV2l0aFJlcXVlc3QgPSBnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShcbiAgICAgICAgY29udGV4dCxcbiAgICAgICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCxcbiAgICAgICAgcmVxdWVzdCxcbiAgICAgICAgZGF0YVNvdXJjZUlkLFxuICAgICAgICB0aGlzLmNsaWVudFxuICAgICAgKTtcbiAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KFxuICAgICAgICAnZm9yZWNhc3QucnVuT25jZUZvcmVjYXN0ZXInLCB7XG4gICAgICAgIGZvcmVjYXN0ZXJJZCxcbiAgICAgIH0pO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgLy8gcmV0dXJuIHRhc2tJZFxuICAgICAgICAgIHJlc3BvbnNlOiByZXNwb25zZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5sb2coJ0ZvcmVjYXN0IC0gcnVuT25jZUZvcmVjYXN0ZXInLCBlcnIpO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBnZXRFcnJvck1lc3NhZ2UoZXJyKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBwdXRGb3JlY2FzdGVyID0gYXN5bmMgKFxuICAgIGNvbnRleHQ6IFJlcXVlc3RIYW5kbGVyQ29udGV4dCxcbiAgICByZXF1ZXN0OiBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gICAgb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZTogT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnlcbiAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnk+PiA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgZm9yZWNhc3RlcklkIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGZvcmVjYXN0ZXJJZDogc3RyaW5nIH07XG4gICAgICBjb25zdCB7IGRhdGFTb3VyY2VJZCA9ICcnIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGRhdGFTb3VyY2VJZD86IHN0cmluZyB9O1xuXG4gICAgICAvL0B0cy1pZ25vcmVcbiAgICAgIGNvbnN0IGlmU2VxTm8gPSByZXF1ZXN0LmJvZHkuc2VxTm87XG4gICAgICAvL0B0cy1pZ25vcmVcbiAgICAgIGNvbnN0IGlmUHJpbWFyeVRlcm0gPSByZXF1ZXN0LmJvZHkucHJpbWFyeVRlcm07XG5cbiAgICAgIGNvbnN0IHJlcXVlc3RCb2R5ID0gSlNPTi5zdHJpbmdpZnkoXG4gICAgICAgIGNvbnZlcnRGb3JlY2FzdEtleXNUb1NuYWtlQ2FzZShyZXF1ZXN0LmJvZHkpXG4gICAgICApO1xuXG4gICAgICBsZXQgcGFyYW1zOiBQdXRGb3JlY2FzdGVyUGFyYW1zID0ge1xuICAgICAgICBmb3JlY2FzdGVySWQ6IGZvcmVjYXN0ZXJJZCxcbiAgICAgICAgaWZTZXFObzogaWZTZXFObyxcbiAgICAgICAgaWZQcmltYXJ5VGVybTogaWZQcmltYXJ5VGVybSxcbiAgICAgICAgYm9keTogcmVxdWVzdEJvZHksXG4gICAgICB9O1xuICAgICAgbGV0IHJlc3BvbnNlO1xuXG4gICAgICBjb25zdCBjYWxsV2l0aFJlcXVlc3QgPSBnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShcbiAgICAgICAgY29udGV4dCxcbiAgICAgICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCxcbiAgICAgICAgcmVxdWVzdCxcbiAgICAgICAgZGF0YVNvdXJjZUlkLFxuICAgICAgICB0aGlzLmNsaWVudFxuICAgICAgKTtcblxuICAgICAgaWYgKGlzTnVtYmVyKGlmU2VxTm8pICYmIGlzTnVtYmVyKGlmUHJpbWFyeVRlcm0pKSB7XG4gICAgICAgIHJlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KCdmb3JlY2FzdC51cGRhdGVGb3JlY2FzdGVyJywgcGFyYW1zKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KCdmb3JlY2FzdC5jcmVhdGVGb3JlY2FzdGVyJywge1xuICAgICAgICAgIGJvZHk6IHBhcmFtcy5ib2R5LFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICAgIGNvbnN0IHJlc3AgPSB7XG4gICAgICAgIC4uLnJlc3BvbnNlLmZvcmVjYXN0ZXIsXG4gICAgICAgIGlkOiByZXNwb25zZS5faWQsXG4gICAgICAgIHByaW1hcnlUZXJtOiByZXNwb25zZS5fcHJpbWFyeV90ZXJtLFxuICAgICAgICBzZXFObzogcmVzcG9uc2UuX3NlcV9ubyxcbiAgICAgIH07XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZTogY29udmVydEZvcmVjYXN0S2V5c1RvQ2FtZWxDYXNlKHJlc3ApIGFzIEZvcmVjYXN0ZXIsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUubG9nKCdGb3JlY2FzdCAtIFB1dEZvcmVjYXN0ZXInLCBlcnIpO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBnZXRFcnJvck1lc3NhZ2UoZXJyKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICB2YWxpZGF0ZUZvcmVjYXN0ZXIgPSBhc3luYyAoXG4gICAgY29udGV4dDogUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlOiBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeVxuICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueT4+ID0+IHtcbiAgICB0cnkge1xuICAgICAgbGV0IHsgdmFsaWRhdGlvblR5cGUgfSA9IHJlcXVlc3QucGFyYW1zIGFzIHtcbiAgICAgICAgdmFsaWRhdGlvblR5cGU6IHN0cmluZztcbiAgICAgIH07XG4gICAgICBjb25zdCB7IGRhdGFTb3VyY2VJZCA9ICcnIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGRhdGFTb3VyY2VJZD86IHN0cmluZyB9O1xuXG4gICAgICBjb25zdCBjYWxsV2l0aFJlcXVlc3QgPSBnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShcbiAgICAgICAgY29udGV4dCxcbiAgICAgICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCxcbiAgICAgICAgcmVxdWVzdCxcbiAgICAgICAgZGF0YVNvdXJjZUlkLFxuICAgICAgICB0aGlzLmNsaWVudFxuICAgICAgKTtcblxuICAgICAgY29uc3QgcmVxdWVzdEJvZHkgPSBKU09OLnN0cmluZ2lmeShcbiAgICAgICAgY29udmVydEZvcmVjYXN0S2V5c1RvU25ha2VDYXNlKHJlcXVlc3QuYm9keSlcbiAgICAgICk7XG4gICAgICBjb25zdCByZXNwb25zZSA9IGF3YWl0IGNhbGxXaXRoUmVxdWVzdChcbiAgICAgICAgJ2ZvcmVjYXN0LnZhbGlkYXRlRm9yZWNhc3RlcicsIHtcbiAgICAgICAgYm9keTogcmVxdWVzdEJvZHksXG4gICAgICAgIHZhbGlkYXRpb25UeXBlOiB2YWxpZGF0aW9uVHlwZSxcbiAgICAgIH0pO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgcmVzcG9uc2U6IHJlc3BvbnNlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBjb25zb2xlLmxvZygnRm9yZWNhc3QgLSB2YWxpZGF0ZUZvcmVjYXN0ZXInLCBlcnIpO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBnZXRFcnJvck1lc3NhZ2UoZXJyKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBzdWdnZXN0Rm9yZWNhc3RlciA9IGFzeW5jIChcbiAgICBjb250ZXh0OiBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gICAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICAgIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55Pj4gPT4ge1xuICAgIHRyeSB7XG4gICAgICBsZXQgeyBzdWdnZXN0VHlwZSB9ID0gcmVxdWVzdC5wYXJhbXMgYXMge1xuICAgICAgICBzdWdnZXN0VHlwZTogc3RyaW5nO1xuICAgICAgfTtcbiAgICAgIGNvbnN0IHsgZGF0YVNvdXJjZUlkID0gJycgfSA9IHJlcXVlc3QucGFyYW1zIGFzIHsgZGF0YVNvdXJjZUlkPzogc3RyaW5nIH07XG5cbiAgICAgIGNvbnN0IGNhbGxXaXRoUmVxdWVzdCA9IGdldENsaWVudEJhc2VkT25EYXRhU291cmNlKFxuICAgICAgICBjb250ZXh0LFxuICAgICAgICB0aGlzLmRhdGFTb3VyY2VFbmFibGVkLFxuICAgICAgICByZXF1ZXN0LFxuICAgICAgICBkYXRhU291cmNlSWQsXG4gICAgICAgIHRoaXMuY2xpZW50XG4gICAgICApO1xuXG4gICAgICBjb25zdCByZXF1ZXN0Qm9keSA9IEpTT04uc3RyaW5naWZ5KFxuICAgICAgICBjb252ZXJ0Rm9yZWNhc3RLZXlzVG9TbmFrZUNhc2UocmVxdWVzdC5ib2R5KVxuICAgICAgKTtcbiAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KFxuICAgICAgICAnZm9yZWNhc3Quc3VnZ2VzdEZvcmVjYXN0ZXInLCB7XG4gICAgICAgIGJvZHk6IHJlcXVlc3RCb2R5LFxuICAgICAgICBzdWdnZXN0VHlwZTogc3VnZ2VzdFR5cGUsXG4gICAgICB9KTtcbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgIHJlc3BvbnNlOiByZXNwb25zZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5sb2coJ0ZvcmVjYXN0IC0gc3VnZ2VzdEZvcmVjYXN0ZXInLCBlcnIpO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBnZXRFcnJvck1lc3NhZ2UoZXJyKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBnZXRGb3JlY2FzdGVyID0gYXN5bmMgKFxuICAgIGNvbnRleHQ6IFJlcXVlc3RIYW5kbGVyQ29udGV4dCxcbiAgICByZXF1ZXN0OiBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gICAgb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZTogT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnlcbiAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnk+PiA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgZm9yZWNhc3RlcklkIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGZvcmVjYXN0ZXJJZDogc3RyaW5nIH07XG4gICAgICBjb25zdCB7IGRhdGFTb3VyY2VJZCA9ICcnIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGRhdGFTb3VyY2VJZD86IHN0cmluZyB9O1xuXG4gICAgICBjb25zdCBjYWxsV2l0aFJlcXVlc3QgPSBnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShcbiAgICAgICAgY29udGV4dCxcbiAgICAgICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCxcbiAgICAgICAgcmVxdWVzdCxcbiAgICAgICAgZGF0YVNvdXJjZUlkLFxuICAgICAgICB0aGlzLmNsaWVudFxuICAgICAgKTtcblxuICAgICAgY29uc3QgZm9yZWNhc3RlclJlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KCdmb3JlY2FzdC5nZXRGb3JlY2FzdGVyJywge1xuICAgICAgICBmb3JlY2FzdGVySWQsXG4gICAgICB9KTtcblxuICAgICAgLy8gUG9wdWxhdGluZyBzdGF0aWMgZm9yZWNhc3RlciBmaWVsZHNcbiAgICAgIGNvbnN0IHN0YXRpY0ZpZWxkcyA9IHtcbiAgICAgICAgaWQ6IGZvcmVjYXN0ZXJSZXNwb25zZS5faWQsXG4gICAgICAgIHByaW1hcnlUZXJtOiBmb3JlY2FzdGVyUmVzcG9uc2UuX3ByaW1hcnlfdGVybSxcbiAgICAgICAgc2VxTm86IGZvcmVjYXN0ZXJSZXNwb25zZS5fc2VxX25vLFxuICAgICAgICAvLyB0aGUgYmFja2VuZCByZXR1cm5zIGEgcmVzcG9uc2Ugd2l0aCBmb3JlY2FzdGVyIGZpZWxkLlxuICAgICAgICAuLi5jb252ZXJ0U3RhdGljRmllbGRzVG9DYW1lbENhc2UoZm9yZWNhc3RlclJlc3BvbnNlLmZvcmVjYXN0ZXIpLFxuICAgICAgfTtcblxuICAgICAgLy8gR2V0IHJlYWwtdGltZSBhbmQgcnVuLW9uY2UgdGFzayBpbmZvIHRvIHBvcHVsYXRlIHRoZVxuICAgICAgLy8gdGFzayBhbmQgam9iLXJlbGF0ZWQgZmllbGRzXG4gICAgICAvLyBXZSB3cmFwIHRoZXNlIGNhbGxzIGluIGEgdHJ5L2NhdGNoLCBhbmQgc3VwcHJlc3MgYW55IGluZGV4X25vdF9mb3VuZF9leGNlcHRpb25zXG4gICAgICAvLyB3aGljaCBjYW4gb2NjdXIgaWYgbm8gZm9yZWNhc3RlciBqb2JzIGhhdmUgYmVlbiByYW4gb24gYSBuZXcgY2x1c3Rlci5cbiAgICAgIC8vIGxldCByZWFsdGltZVRhc2tzUmVzcG9uc2UgPSB7fSBhcyBhbnk7XG4gICAgICAvLyBsZXQgcnVuT25jZVRhc2tzUmVzcG9uc2UgPSB7fSBhcyBhbnk7XG4gICAgICAvLyB0cnkge1xuICAgICAgLy8gICBjb25zdCBjYWxsV2l0aFJlcXVlc3QgPSBnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShcbiAgICAgIC8vICAgICBjb250ZXh0LFxuICAgICAgLy8gICAgIHRoaXMuZGF0YVNvdXJjZUVuYWJsZWQsXG4gICAgICAvLyAgICAgcmVxdWVzdCxcbiAgICAgIC8vICAgICBkYXRhU291cmNlSWQsXG4gICAgICAvLyAgICAgdGhpcy5jbGllbnRcbiAgICAgIC8vICAgKTtcblxuICAgICAgLy8gICByZWFsdGltZVRhc2tzUmVzcG9uc2UgPSBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoJ2ZvcmVjYXN0LnNlYXJjaFRhc2tzJywge1xuICAgICAgLy8gICAgIGJvZHk6IGdldExhdGVzdFRhc2tGb3JGb3JlY2FzdGVyUXVlcnkoZm9yZWNhc3RlcklkLCB0cnVlKSxcbiAgICAgIC8vICAgfSk7XG5cbiAgICAgIC8vICAgcnVuT25jZVRhc2tzUmVzcG9uc2UgPSBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoJ2ZvcmVjYXN0LnNlYXJjaFRhc2tzJywge1xuICAgICAgLy8gICAgIGJvZHk6IGdldExhdGVzdFRhc2tGb3JGb3JlY2FzdGVyUXVlcnkoZm9yZWNhc3RlcklkLCBmYWxzZSksXG4gICAgICAvLyAgIH0pO1xuICAgICAgLy8gfSBjYXRjaCAoZXJyKSB7XG4gICAgICAvLyAgIGlmICghaXNJbmRleE5vdEZvdW5kRXJyb3IoZXJyKSkge1xuICAgICAgLy8gICAgIHRocm93IGVycjtcbiAgICAgIC8vICAgfVxuICAgICAgLy8gfVxuXG4gICAgICAvLyBjb25zdCByZWFsdGltZVRhc2sgPSBnZXQoXG4gICAgICAvLyAgIGdldChyZWFsdGltZVRhc2tzUmVzcG9uc2UsICdoaXRzLmhpdHMnLCBbXSkubWFwKCh0YXNrUmVzcG9uc2U6IGFueSkgPT4ge1xuICAgICAgLy8gICAgIHJldHVybiB7XG4gICAgICAvLyAgICAgICBpZDogZ2V0KHRhc2tSZXNwb25zZSwgJ19pZCcpLFxuICAgICAgLy8gICAgICAgLi4uZ2V0KHRhc2tSZXNwb25zZSwgJ19zb3VyY2UnKSxcbiAgICAgIC8vICAgICB9O1xuICAgICAgLy8gICB9KSxcbiAgICAgIC8vICAgMFxuICAgICAgLy8gKTtcbiAgICAgIC8vIGNvbnN0IHJ1bk9uY2VUYXNrID0gZ2V0KFxuICAgICAgLy8gICBnZXQocnVuT25jZVRhc2tzUmVzcG9uc2UsICdoaXRzLmhpdHMnLCBbXSkubWFwKFxuICAgICAgLy8gICAgICh0YXNrUmVzcG9uc2U6IGFueSkgPT4ge1xuICAgICAgLy8gICAgICAgcmV0dXJuIHtcbiAgICAgIC8vICAgICAgICAgaWQ6IGdldCh0YXNrUmVzcG9uc2UsICdfaWQnKSxcbiAgICAgIC8vICAgICAgICAgLi4uZ2V0KHRhc2tSZXNwb25zZSwgJ19zb3VyY2UnKSxcbiAgICAgIC8vICAgICAgIH07XG4gICAgICAvLyAgICAgfVxuICAgICAgLy8gICApLFxuICAgICAgLy8gICAwXG4gICAgICAvLyApO1xuXG4gICAgICBjb25zdCB0YXNrQW5kSm9iRmllbGRzID0gY29udmVydFRhc2tBbmRKb2JGaWVsZHNUb0NhbWVsQ2FzZShcbiAgICAgICAgZm9yZWNhc3RlclJlc3BvbnNlLnJlYWx0aW1lX3Rhc2ssXG4gICAgICAgIGZvcmVjYXN0ZXJSZXNwb25zZS5ydW5fb25jZV90YXNrLFxuICAgICAgICBmb3JlY2FzdGVyUmVzcG9uc2UuZm9yZWNhc3Rlcl9qb2JcbiAgICAgICk7XG5cbiAgICAgIC8vIENvbWJpbmUgdGhlIHN0YXRpYyBhbmQgdGFzay1hbmQtam9iLXJlbGF0ZWQgZmllbGRzIGludG9cbiAgICAgIC8vIGEgZmluYWwgcmVzcG9uc2VcbiAgICAgIGNvbnN0IGZpbmFsUmVzcG9uc2UgPSB7XG4gICAgICAgIC4uLnN0YXRpY0ZpZWxkcyxcbiAgICAgICAgLi4udGFza0FuZEpvYkZpZWxkcyxcbiAgICAgIH07XG5cbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgIHJlc3BvbnNlOiBmaW5hbFJlc3BvbnNlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICAvLyBpZiB0aGUgZm9yZWNhc3RlciBpcyBub3QgZm91bmQgKGUuZy4sIGRlbGV0ZWQgd2hpbGUgb24gdGhlIGRldGFpbCBwYWdlKSwgcmV0dXJuIGFuIGVtcHR5IHJlc3BvbnNlXG4gICAgICAvLyB0aGlzIGlzIHRvIGF2b2lkIHRoZSBlcnJvciBtZXNzYWdlIGZyb20gdGhlIGZyb250ZW5kIHdoZXJlIHRoZSBmb3JlY2FzdGVyIGlzIG5vdCBmb3VuZFxuICAgICAgLy8gdGhlIGVycm9yIGlzIHRyaWdnZXJlZCBieSB1c2VFZmZlY3Qgb2YgdXNlRmV0Y2hGb3JlY2FzdGVySW5mbyBpbiBGb3JlY2FzdGVyRGV0YWlsXG4gICAgICBpZiAoXG4gICAgICAgIGVyci5zdGF0dXNDb2RlID09PSA0MDRcbiAgICAgICkge1xuICAgICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keTogeyBvazogdHJ1ZSwgcmVzcG9uc2U6IHt9IH0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgICAgY29uc29sZS5sb2coJ0ZvcmVjYXN0IC0gVW5hYmxlIHRvIGdldCBmb3JlY2FzdGVyJywgZXJyKTtcbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICBlcnJvcjogZ2V0RXJyb3JNZXNzYWdlKGVyciksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgc3RhcnRGb3JlY2FzdGVyID0gYXN5bmMgKFxuICAgIGNvbnRleHQ6IFJlcXVlc3RIYW5kbGVyQ29udGV4dCxcbiAgICByZXF1ZXN0OiBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gICAgb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZTogT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnlcbiAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnk+PiA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHsgZm9yZWNhc3RlcklkIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGZvcmVjYXN0ZXJJZDogc3RyaW5nIH07XG4gICAgICBjb25zdCB7IGRhdGFTb3VyY2VJZCA9ICcnIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGRhdGFTb3VyY2VJZD86IHN0cmluZyB9O1xuICAgICAgLy9AdHMtaWdub3JlXG4gICAgICBjb25zdCBzdGFydFRpbWUgPSByZXF1ZXN0LmJvZHk/LnN0YXJ0VGltZTtcbiAgICAgIC8vQHRzLWlnbm9yZVxuICAgICAgY29uc3QgZW5kVGltZSA9IHJlcXVlc3QuYm9keT8uZW5kVGltZTtcbiAgICAgIGxldCByZXF1ZXN0UGFyYW1zID0geyBmb3JlY2FzdGVySWQ6IGZvcmVjYXN0ZXJJZCB9IGFzIHt9O1xuICAgICAgbGV0IHJlcXVlc3RQYXRoID0gJ2ZvcmVjYXN0LnN0YXJ0Rm9yZWNhc3Rlcic7XG5cbiAgICAgIGNvbnN0IGNhbGxXaXRoUmVxdWVzdCA9IGdldENsaWVudEJhc2VkT25EYXRhU291cmNlKFxuICAgICAgICBjb250ZXh0LFxuICAgICAgICB0aGlzLmRhdGFTb3VyY2VFbmFibGVkLFxuICAgICAgICByZXF1ZXN0LFxuICAgICAgICBkYXRhU291cmNlSWQsXG4gICAgICAgIHRoaXMuY2xpZW50XG4gICAgICApO1xuXG4gICAgICBjb25zdCByZXNwb25zZSA9IGF3YWl0IGNhbGxXaXRoUmVxdWVzdChyZXF1ZXN0UGF0aCwgcmVxdWVzdFBhcmFtcyk7XG5cbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgIHJlc3BvbnNlOiByZXNwb25zZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5sb2coJ0ZvcmVjYXN0IC0gc3RhcnRGb3JlY2FzdGVyJywgZXJyKTtcbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICBlcnJvcjogZ2V0RXJyb3JNZXNzYWdlKGVyciksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgc3RvcEZvcmVjYXN0ZXIgPSBhc3luYyAoXG4gICAgY29udGV4dDogUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlOiBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeVxuICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueT4+ID0+IHtcbiAgICB0cnkge1xuICAgICAgLy8gRXh0cmFjdCByZXF1aXJlZCBwYXJhbWV0ZXJzIHdpdGggc3BlY2lmaWMgdHlwZSBhc3NlcnRpb24uXG4gICAgICAvLyAnZm9yZWNhc3RlcklkJyBpcyBleHBlY3RlZCB0byBhbHdheXMgYmUgcHJlc2VudCBpbiB0aGUgcm91dGUgcGF0aC5cbiAgICAgIGxldCB7IGZvcmVjYXN0ZXJJZCB9ID0gcmVxdWVzdC5wYXJhbXMgYXMge1xuICAgICAgICBmb3JlY2FzdGVySWQ6IHN0cmluZztcbiAgICAgIH07XG4gICAgICAvLyBFeHRyYWN0IG9wdGlvbmFsIHBhcmFtZXRlcnMgc2VwYXJhdGVseS5cbiAgICAgIC8vICdkYXRhU291cmNlSWQnIG1pZ2h0IGJlIG1pc3NpbmcgZnJvbSB0aGUgcm91dGUgcGF0aCAoaGVuY2UgJz8nKS5cbiAgICAgIC8vIFByb3ZpZGUgYSBkZWZhdWx0IHZhbHVlICgnJykgaWYgaXQncyBub3QgcHJlc2VudCB1c2luZyBkZXN0cnVjdHVyaW5nIGRlZmF1bHQgYXNzaWdubWVudC5cbiAgICAgIGNvbnN0IHsgZGF0YVNvdXJjZUlkID0gJycgfSA9IHJlcXVlc3QucGFyYW1zIGFzIHsgZGF0YVNvdXJjZUlkPzogc3RyaW5nIH07XG5cbiAgICAgIGNvbnN0IHJlcXVlc3RQYXRoID0gJ2ZvcmVjYXN0LnN0b3BGb3JlY2FzdGVyJztcblxuICAgICAgY29uc3QgY2FsbFdpdGhSZXF1ZXN0ID0gZ2V0Q2xpZW50QmFzZWRPbkRhdGFTb3VyY2UoXG4gICAgICAgIGNvbnRleHQsXG4gICAgICAgIHRoaXMuZGF0YVNvdXJjZUVuYWJsZWQsXG4gICAgICAgIHJlcXVlc3QsXG4gICAgICAgIGRhdGFTb3VyY2VJZCxcbiAgICAgICAgdGhpcy5jbGllbnRcbiAgICAgICk7XG5cbiAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KHJlcXVlc3RQYXRoLCB7XG4gICAgICAgIGZvcmVjYXN0ZXJJZDogZm9yZWNhc3RlcklkLFxuICAgICAgfSk7XG5cbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgIHJlc3BvbnNlOiByZXNwb25zZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5sb2coJ0ZvcmVjYXN0IC0gc3RvcEZvcmVjYXN0ZXInLCBlcnIpO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBnZXRFcnJvck1lc3NhZ2UoZXJyKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBnZXRGb3JlY2FzdGVyUHJvZmlsZSA9IGFzeW5jIChcbiAgICBjb250ZXh0OiBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gICAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICAgIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55Pj4gPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCB7IGZvcmVjYXN0ZXJJZCB9ID0gcmVxdWVzdC5wYXJhbXMgYXMgeyBmb3JlY2FzdGVySWQ6IHN0cmluZyB9O1xuICAgICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCB0aGlzLmNsaWVudFxuICAgICAgICAuYXNTY29wZWQocmVxdWVzdClcbiAgICAgICAgLmNhbGxBc0N1cnJlbnRVc2VyKCdmb3JlY2FzdC5mb3JlY2FzdGVyUHJvZmlsZScsIHtcbiAgICAgICAgICBmb3JlY2FzdGVySWQsXG4gICAgICAgIH0pO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgcmVzcG9uc2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUubG9nKCdGb3JlY2FzdCAtIGZvcmVjYXN0ZXJQcm9maWxlJywgZXJyKTtcbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICBlcnJvcjogZ2V0RXJyb3JNZXNzYWdlKGVyciksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG5cbiAgc2VhcmNoRm9yZWNhc3RlciA9IGFzeW5jIChcbiAgICBjb250ZXh0OiBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gICAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICAgIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55Pj4gPT4ge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCByZXF1ZXN0Qm9keSA9IEpTT04uc3RyaW5naWZ5KHJlcXVlc3QuYm9keSk7XG4gICAgICBjb25zdCByZXNwb25zZTogU2VhcmNoUmVzcG9uc2U8Rm9yZWNhc3Rlcj4gPSBhd2FpdCB0aGlzLmNsaWVudFxuICAgICAgICAuYXNTY29wZWQocmVxdWVzdClcbiAgICAgICAgLmNhbGxBc0N1cnJlbnRVc2VyKCdmb3JlY2FzdC5zZWFyY2hGb3JlY2FzdGVyJywgeyBib2R5OiByZXF1ZXN0Qm9keSB9KTtcbiAgICAgIGNvbnN0IHRvdGFsRm9yZWNhc3RlcnMgPSBnZXQocmVzcG9uc2UsICdoaXRzLnRvdGFsLnZhbHVlJywgMCk7XG4gICAgICBjb25zdCBmb3JlY2FzdGVycyA9IGdldChyZXNwb25zZSwgJ2hpdHMuaGl0cycsIFtdKS5tYXAoKGZvcmVjYXN0ZXI6IGFueSkgPT4gKHtcbiAgICAgICAgLi4uY29udmVydEZvcmVjYXN0S2V5c1RvQ2FtZWxDYXNlKGZvcmVjYXN0ZXIuX3NvdXJjZSksXG4gICAgICAgIGlkOiBmb3JlY2FzdGVyLl9pZCxcbiAgICAgICAgc2VxTm86IGZvcmVjYXN0ZXIuX3NlcV9ubyxcbiAgICAgICAgcHJpbWFyeVRlcm06IGZvcmVjYXN0ZXIuX3ByaW1hcnlfdGVybSxcbiAgICAgIH0pKTtcbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgIHJlc3BvbnNlOiB7XG4gICAgICAgICAgICB0b3RhbEZvcmVjYXN0ZXJzLFxuICAgICAgICAgICAgZm9yZWNhc3RlcnMsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgY29uc29sZS5sb2coJ0ZvcmVjYXN0IC0gVW5hYmxlIHRvIHNlYXJjaCBmb3JlY2FzdGVycycsIGVycik7XG4gICAgICBpZiAoaXNJbmRleE5vdEZvdW5kRXJyb3IoZXJyKSkge1xuICAgICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keTogeyBvazogdHJ1ZSwgcmVzcG9uc2U6IHsgdG90YWxGb3JlY2FzdGVyczogMCwgZm9yZWNhc3RlcnM6IFtdIH0gfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgZXJyb3I6IGdldEVycm9yTWVzc2FnZShlcnIpLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICB9O1xuXG4gIGdldEZvcmVjYXN0ZXJzID0gYXN5bmMgKFxuICAgIGNvbnRleHQ6IFJlcXVlc3RIYW5kbGVyQ29udGV4dCxcbiAgICByZXF1ZXN0OiBPcGVuU2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QsXG4gICAgb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZTogT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZUZhY3RvcnlcbiAgKTogUHJvbWlzZTxJT3BlblNlYXJjaERhc2hib2FyZHNSZXNwb25zZTxhbnk+PiA9PiB7XG4gICAgdHJ5IHtcbiAgICAgIGNvbnN0IHtcbiAgICAgICAgZGF0YVNvdXJjZUlkID0gJycsXG4gICAgICB9ID0gcmVxdWVzdC5wYXJhbXMgYXMgeyBkYXRhU291cmNlSWQ/OiBzdHJpbmcgfTtcblxuICAgICAgY29uc3QgY2FsbFdpdGhSZXF1ZXN0ID0gZ2V0Q2xpZW50QmFzZWRPbkRhdGFTb3VyY2UoXG4gICAgICAgIGNvbnRleHQsXG4gICAgICAgIHRoaXMuZGF0YVNvdXJjZUVuYWJsZWQsXG4gICAgICAgIHJlcXVlc3QsXG4gICAgICAgIGRhdGFTb3VyY2VJZCxcbiAgICAgICAgdGhpcy5jbGllbnRcbiAgICAgICk7XG4gICAgICBjb25zdCByZXNwb25zZSA9IGF3YWl0IGNhbGxXaXRoUmVxdWVzdCgnZm9yZWNhc3Quc2VhcmNoRm9yZWNhc3RlcicsIHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIHNpemU6IE1BWF9GT1JFQ0FTVEVSLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgICBjb25zdCB0b3RhbEZvcmVjYXN0ZXJzID0gZ2V0KHJlc3BvbnNlLCAnaGl0cy50b3RhbC52YWx1ZScsIDApO1xuXG4gICAgICAvL0dldCBhbGwgZm9yZWNhc3RlcnMgZnJvbSBzZWFyY2ggZm9yZWNhc3RlciBBUElcbiAgICAgIGNvbnN0IGFsbEZvcmVjYXN0ZXJzID0gZ2V0KHJlc3BvbnNlLCAnaGl0cy5oaXRzJywgW10pLnJlZHVjZShcbiAgICAgICAgKGFjYzogYW55LCBmb3JlY2FzdGVyUmVzcG9uc2U6IGFueSkgPT4gKHtcbiAgICAgICAgICAuLi5hY2MsXG4gICAgICAgICAgW2ZvcmVjYXN0ZXJSZXNwb25zZS5faWRdOiB7XG4gICAgICAgICAgICBpZDogZm9yZWNhc3RlclJlc3BvbnNlLl9pZCxcbiAgICAgICAgICAgIHByaW1hcnlUZXJtOiBmb3JlY2FzdGVyUmVzcG9uc2UuX3ByaW1hcnlfdGVybSxcbiAgICAgICAgICAgIHNlcU5vOiBmb3JlY2FzdGVyUmVzcG9uc2UuX3NlcV9ubyxcbiAgICAgICAgICAgIC4uLmNvbnZlcnRTdGF0aWNGaWVsZHNUb0NhbWVsQ2FzZShmb3JlY2FzdGVyUmVzcG9uc2UuX3NvdXJjZSksXG4gICAgICAgICAgfSxcbiAgICAgICAgfSksXG4gICAgICAgIHt9XG4gICAgICApO1xuXG4gICAgICAvLyBGZXRjaCB0aGUgbGF0ZXN0IHJlYWx0aW1lIGFuZCBydW5PbmNlIHRhc2tzIGZvciBhbGwgZm9yZWNhc3RlcnNcbiAgICAgIC8vIHVzaW5nIHRlcm1zIGFnZ3JlZ2F0aW9uc1xuICAgICAgLy8gV2Ugd3JhcCB0aGVzZSBjYWxscyBpbiBhIHRyeS9jYXRjaCwgYW5kIHN1cHByZXNzIGFueSBpbmRleF9ub3RfZm91bmRfZXhjZXB0aW9uc1xuICAgICAgLy8gd2hpY2ggY2FuIG9jY3VyIGlmIG5vIGZvcmVjYXN0ZXIgam9icyBoYXZlIGJlZW4gcmFuIG9uIGEgbmV3IGNsdXN0ZXIuXG4gICAgICBsZXQgcmVhbHRpbWVUYXNrc1Jlc3BvbnNlID0ge30gYXMgYW55O1xuICAgICAgbGV0IHJ1bk9uY2VUYXNrc1Jlc3BvbnNlID0ge30gYXMgYW55O1xuICAgICAgdHJ5IHtcbiAgICAgICAgcmVhbHRpbWVUYXNrc1Jlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KCdmb3JlY2FzdC5zZWFyY2hUYXNrcycsIHtcbiAgICAgICAgICBib2R5OiBnZXRMYXRlc3RGb3JlY2FzdGVyVGFza3NRdWVyeSh0cnVlKSxcbiAgICAgICAgfSk7XG4gICAgICAgIHJ1bk9uY2VUYXNrc1Jlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KCdmb3JlY2FzdC5zZWFyY2hUYXNrcycsIHtcbiAgICAgICAgICBib2R5OiBnZXRMYXRlc3RGb3JlY2FzdGVyVGFza3NRdWVyeShmYWxzZSksXG4gICAgICAgIH0pO1xuICAgICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICAgIGlmICghaXNJbmRleE5vdEZvdW5kRXJyb3IoZXJyKSkge1xuICAgICAgICAgIHRocm93IGVycjtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICBjb25zdCByZWFsdGltZVRhc2tzID0gZ2V0KFxuICAgICAgICByZWFsdGltZVRhc2tzUmVzcG9uc2UsXG4gICAgICAgICdhZ2dyZWdhdGlvbnMuZm9yZWNhc3RlcnMuYnVja2V0cycsXG4gICAgICAgIFtdXG4gICAgICApLnJlZHVjZSgoYWNjOiBhbnksIGJ1Y2tldDogYW55KSA9PiB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgLi4uYWNjLFxuICAgICAgICAgIFtidWNrZXQua2V5XToge1xuICAgICAgICAgICAgcmVhbHRpbWVUYXNrOiBnZXQoYnVja2V0LCAnbGF0ZXN0X3Rhc2tzLmhpdHMuaGl0cy4wJywgdW5kZWZpbmVkKSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgICAgfSwge30pO1xuXG4gICAgICBjb25zdCBydW5PbmNlVGFza3MgPSBnZXQoXG4gICAgICAgIHJ1bk9uY2VUYXNrc1Jlc3BvbnNlLFxuICAgICAgICAnYWdncmVnYXRpb25zLmZvcmVjYXN0ZXJzLmJ1Y2tldHMnLFxuICAgICAgICBbXVxuICAgICAgKS5yZWR1Y2UoKGFjYzogYW55LCBidWNrZXQ6IGFueSkgPT4ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIC4uLmFjYyxcbiAgICAgICAgICBbYnVja2V0LmtleV06IHtcbiAgICAgICAgICAgIHJ1bk9uY2VUYXNrOiBnZXQoYnVja2V0LCAnbGF0ZXN0X3Rhc2tzLmhpdHMuaGl0cy4wJywgdW5kZWZpbmVkKSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgICAgfSwge30pO1xuXG4gICAgICAvLyBHZXQgcmVhbC10aW1lIGFuZCBydW5PbmNlIHRhc2sgaW5mbyBieSBsb29waW5nIHRocm91Z2ggZWFjaCBmb3JlY2FzdGVyICYgcmV0cmlldmluZ1xuICAgICAgLy8gICAgLSBjdXJTdGF0ZSBieSBnZXR0aW5nIHJlYWwtdGltZSB0YXNrIHN0YXRlXG4gICAgICAvLyAgICAtIGVuYWJsZWRUaW1lIGJ5IGdldHRpbmcgcmVhbC10aW1lIHRhc2sncyBleGVjdXRpb25fc3RhcnQgdGltZVxuICAgICAgLy8gICAgLSB0YXNrSWQgYnkgZ2V0dGluZyBoaXN0b3JpY2FsIHRhc2sncyBfaWRcbiAgICAgIGNvbnN0IGZvcmVjYXN0ZXJzQXJyYXkgPSBPYmplY3QudmFsdWVzKGFsbEZvcmVjYXN0ZXJzKTtcbiAgICAgIGZvcmVjYXN0ZXJzQXJyYXkuZm9yRWFjaCgoZm9yZWNhc3RlcjogYW55KSA9PiB7XG4gICAgICAgIGNvbnN0IHJlYWx0aW1lVGFzayA9IGdldChcbiAgICAgICAgICByZWFsdGltZVRhc2tzW2ZvcmVjYXN0ZXIuaWRdLFxuICAgICAgICAgICdyZWFsdGltZVRhc2suX3NvdXJjZSdcbiAgICAgICAgKTtcbiAgICAgICAgY29uc3QgcnVuT25jZVRhc2sgPSBnZXQoXG4gICAgICAgICAgcnVuT25jZVRhc2tzW2ZvcmVjYXN0ZXIuaWRdLFxuICAgICAgICAgICdydW5PbmNlVGFzay5fc291cmNlJ1xuICAgICAgICApO1xuXG4gICAgICAgIGZvcmVjYXN0ZXIuY3VyU3RhdGUgPSBjb21iaW5lVGFza1N0YXRlKHJlYWx0aW1lVGFzaywgcnVuT25jZVRhc2spO1xuICAgICAgICBmb3JlY2FzdGVyLnJlYWxUaW1lTGFzdFVwZGF0ZVRpbWUgPSBnZXQocmVhbHRpbWVUYXNrLCAnbGFzdF91cGRhdGVfdGltZScpO1xuICAgICAgICBmb3JlY2FzdGVyLnJ1bk9uY2VMYXN0VXBkYXRlVGltZSA9IGdldChydW5PbmNlVGFzaywgJ2xhc3RfdXBkYXRlX3RpbWUnKTtcbiAgICAgICAgZm9yZWNhc3Rlci5zdGF0ZUVycm9yID0gZ2V0KHJlYWx0aW1lVGFzaywgJ2Vycm9yJykgfHwgZ2V0KHJ1bk9uY2VUYXNrLCAnZXJyb3InKTtcbiAgICAgIH0pO1xuXG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZToge1xuICAgICAgICAgICAgdG90YWxGb3JlY2FzdGVyczogdG90YWxGb3JlY2FzdGVycyxcbiAgICAgICAgICAgIGZvcmVjYXN0ZXJMaXN0OiBmb3JlY2FzdGVyc0FycmF5LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUubG9nKCdGb3JlY2FzdGVyIC0gVW5hYmxlIHRvIHNlYXJjaCBmb3JlY2FzdGVycycsIGVycik7XG4gICAgICBpZiAoaXNJbmRleE5vdEZvdW5kRXJyb3IoZXJyKSkge1xuICAgICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgICAgYm9keTogeyBvazogdHJ1ZSwgcmVzcG9uc2U6IHsgdG90YWxGb3JlY2FzdGVyczogMCwgZm9yZWNhc3Rlckxpc3Q6IFtdIH0gfSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogZmFsc2UsXG4gICAgICAgICAgZXJyb3I6IGdldEVycm9yTWVzc2FnZShlcnIpLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfVxuICB9O1xuXG4gIGdldEZvcmVjYXN0UmVzdWx0cyA9IGFzeW5jIChcbiAgICBjb250ZXh0OiBSZXF1ZXN0SGFuZGxlckNvbnRleHQsXG4gICAgcmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxuICAgIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2VGYWN0b3J5XG4gICk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U8YW55Pj4gPT4ge1xuICAgIGxldCB7IGlkLCBpc1J1bk9uY2UsIHJlc3VsdEluZGV4LCAgfSA9XG4gICAgICByZXF1ZXN0LnBhcmFtcyBhcyB7XG4gICAgICAgIGlkOiBzdHJpbmc7XG4gICAgICAgIGlzUnVuT25jZTogYW55O1xuICAgICAgICByZXN1bHRJbmRleDogc3RyaW5nO1xuICAgICAgfTtcbiAgICBjb25zdCB7IGRhdGFTb3VyY2VJZCA9ICcnIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGRhdGFTb3VyY2VJZD86IHN0cmluZyB9O1xuXG4gICAgaWYgKCFyZXN1bHRJbmRleCkge1xuICAgICAgLy8gTm90IHN0cmljdGx5IHF1ZXJ5aW5nIGN1c3RvbSDih5IgZGVmYXVsdCB0byAnJ1xuICAgICAgcmVzdWx0SW5kZXggPSAnJztcbiAgICB9IGVsc2UgaWYgKCFyZXN1bHRJbmRleC5zdGFydHNXaXRoKENVU1RPTV9GT1JFQ0FTVF9SRVNVTFRfSU5ERVhfUFJFRklYKSkge1xuICAgICAgLy8gSWYgcmVzdWx0SW5kZXggaXMgZ2l2ZW4gYnV0IG5vdCB2YWxpZCwgcmV2ZXJ0IHRvIGRlZmF1bHRcbiAgICAgIHJlc3VsdEluZGV4ID0gJyc7XG4gICAgfVxuXG4gICAgaXNSdW5PbmNlID0gSlNPTi5wYXJzZShpc1J1bk9uY2UpO1xuXG4gICAgLy8gU2VhcmNoIGJ5IHRhc2sgaWQgaWYgcnVuT25jZSwgb3IgYnkgZm9yZWNhc3RlciBpZCBpZiByZWFsdGltZVxuICAgIGNvbnN0IHNlYXJjaFRlcm0gPSBpc1J1bk9uY2UgPyB7IHRhc2tfaWQ6IGlkIH0gOiB7IGZvcmVjYXN0ZXJfaWQ6IGlkIH07XG5cbiAgICB0cnkge1xuICAgICAgY29uc3Qge1xuICAgICAgICBzaXplID0gMjAsXG4gICAgICAgIHNvcnREaXJlY3Rpb24gPSBTT1JUX0RJUkVDVElPTi5ERVNDLFxuICAgICAgICBzb3J0RmllbGQgPSBGT1JFQ0FTVEVSX0RPQ19GSUVMRFMuREFUQV9TVEFSVF9USU1FLFxuICAgICAgICBzdGFydFRpbWUgPSAwLFxuICAgICAgICBlbmRUaW1lID0gMCxcbiAgICAgICAgZmllbGROYW1lID0gJycsXG4gICAgICAgIGVudGl0eUxpc3QgPSAnJyxcbiAgICAgICAgZGF3bkVwb2NoID0gMCxcbiAgICAgICAgbWF4RW50aXRpZXMgPSAwLFxuICAgICAgfSA9IHJlcXVlc3QucXVlcnkgYXMge1xuICAgICAgICBzaXplOiBudW1iZXI7XG4gICAgICAgIHNvcnREaXJlY3Rpb246IFNPUlRfRElSRUNUSU9OO1xuICAgICAgICBzb3J0RmllbGQ/OiBzdHJpbmc7XG4gICAgICAgIHN0YXJ0VGltZTogbnVtYmVyO1xuICAgICAgICBlbmRUaW1lOiBudW1iZXI7XG4gICAgICAgIGZpZWxkTmFtZTogc3RyaW5nO1xuICAgICAgICBlbnRpdHlMaXN0OiBzdHJpbmc7XG4gICAgICAgIGRhd25FcG9jaDogbnVtYmVyO1xuICAgICAgICBtYXhFbnRpdGllcz86IG51bWJlcjsgLy8gd2UgZGVmYXVsdCB0byAwIGlmIG5vdCBwcm92aWRlZFxuICAgICAgfTtcblxuICAgICAgLy9BbGxvd2VkIHNvcnRpbmcgY29sdW1uc1xuICAgICAgY29uc3Qgc29ydFF1ZXJ5TWFwID0ge1xuICAgICAgICBbRk9SRUNBU1RFUl9ET0NfRklFTERTLkRBVEFfU1RBUlRfVElNRV06IHtcbiAgICAgICAgICBbRk9SRUNBU1RFUl9ET0NfRklFTERTLkRBVEFfU1RBUlRfVElNRV06IHNvcnREaXJlY3Rpb24sXG4gICAgICAgIH0sXG4gICAgICAgIFtGT1JFQ0FTVEVSX0RPQ19GSUVMRFMuREFUQV9FTkRfVElNRV06IHtcbiAgICAgICAgICBbRk9SRUNBU1RFUl9ET0NfRklFTERTLkRBVEFfRU5EX1RJTUVdOiBzb3J0RGlyZWN0aW9uLFxuICAgICAgICB9LFxuICAgICAgfSBhcyB7IFtrZXk6IHN0cmluZ106IG9iamVjdCB9O1xuICAgICAgbGV0IHNvcnQgPSB7fTtcbiAgICAgIGNvbnN0IHNvcnRRdWVyeSA9IHNvcnRRdWVyeU1hcFtzb3J0RmllbGRdO1xuICAgICAgaWYgKHNvcnRRdWVyeSkge1xuICAgICAgICBzb3J0ID0gc29ydFF1ZXJ5O1xuICAgICAgfVxuXG4gICAgICAvL1ByZXBhcmluZyBzZWFyY2ggcmVxdWVzdFxuICAgICAgY29uc3QgcmVxdWVzdEJvZHkgPSB7XG4gICAgICAgIHNvcnQsXG4gICAgICAgIHNpemUsXG4gICAgICAgIHF1ZXJ5OiB7XG4gICAgICAgICAgYm9vbDoge1xuICAgICAgICAgICAgZmlsdGVyOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICB0ZXJtOiBzZWFyY2hUZXJtLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfTtcblxuICAgICAgLy8gSWYgcXVlcnlpbmcgUlQgcmVzdWx0czogcmVtb3ZlIGFueSByZXN1bHRzIHRoYXQgaW5jbHVkZSBhIHRhc2tfaWQsIGFzIHRoaXMgaW5kaWNhdGVzXG4gICAgICAvLyBhIHJ1bk9uY2UgcmVzdWx0IGZyb20gYSBydW5PbmNlIHRhc2suXG4gICAgICBpZiAoIWlzUnVuT25jZSkge1xuICAgICAgICByZXF1ZXN0Qm9keS5xdWVyeS5ib29sID0ge1xuICAgICAgICAgIC4uLnJlcXVlc3RCb2R5LnF1ZXJ5LmJvb2wsXG4gICAgICAgICAgLi4ue1xuICAgICAgICAgICAgbXVzdF9ub3Q6IHtcbiAgICAgICAgICAgICAgZXhpc3RzOiB7XG4gICAgICAgICAgICAgICAgZmllbGQ6ICd0YXNrX2lkJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgIH1cblxuICAgICAgdHJ5IHtcbiAgICAgICAgLy8gR2V0IGN1cnJlbnQgbnVtYmVyIG9mIGZpbHRlcnMgdG8gZGV0ZXJtaW5lIHRoZSBpbmRleCBmb3IgYWRkaW5nIG5ldyBkYXRlIHJhbmdlIGZpbHRlclxuICAgICAgICAvLyBUaGlzIGluY2x1ZGVzIHRoZSBpbml0aWFsIHRlcm0gZmlsdGVyIGFuZCBhbnkgZW50aXR5IGZpbHRlcnMgdGhhdCB3ZXJlIGFkZGVkXG4gICAgICAgIGxldCBmaWx0ZXJTaXplID0gcmVxdWVzdEJvZHkucXVlcnkuYm9vbC5maWx0ZXIubGVuZ3RoO1xuICAgICAgICBpZiAoZmllbGROYW1lKSB7XG4gICAgICAgICAgKHN0YXJ0VGltZSB8fCBlbmRUaW1lKSAmJlxuICAgICAgICAgICAgc2V0KFxuICAgICAgICAgICAgICByZXF1ZXN0Qm9keS5xdWVyeS5ib29sLmZpbHRlcixcbiAgICAgICAgICAgICAgYCR7ZmlsdGVyU2l6ZX0ucmFuZ2UuJHtmaWVsZE5hbWV9LmZvcm1hdGAsXG4gICAgICAgICAgICAgICdlcG9jaF9taWxsaXMnXG4gICAgICAgICAgICApO1xuXG4gICAgICAgICAgc3RhcnRUaW1lICYmXG4gICAgICAgICAgICBzZXQoXG4gICAgICAgICAgICAgIHJlcXVlc3RCb2R5LnF1ZXJ5LmJvb2wuZmlsdGVyLFxuICAgICAgICAgICAgICBgJHtmaWx0ZXJTaXplfS5yYW5nZS4ke2ZpZWxkTmFtZX0uZ3RlYCxcbiAgICAgICAgICAgICAgc3RhcnRUaW1lXG4gICAgICAgICAgICApO1xuXG4gICAgICAgICAgZW5kVGltZSAmJlxuICAgICAgICAgICAgc2V0KFxuICAgICAgICAgICAgICByZXF1ZXN0Qm9keS5xdWVyeS5ib29sLmZpbHRlcixcbiAgICAgICAgICAgICAgYCR7ZmlsdGVyU2l6ZX0ucmFuZ2UuJHtmaWVsZE5hbWV9Lmx0ZWAsXG4gICAgICAgICAgICAgIGVuZFRpbWVcbiAgICAgICAgICAgICk7XG4gICAgICAgIH1cblxuICAgICAgICBmaWx0ZXJTaXplID0gcmVxdWVzdEJvZHkucXVlcnkuYm9vbC5maWx0ZXIubGVuZ3RoO1xuXG4gICAgICAgIC8vIEFkZCBkYXduRXBvY2ggZmlsdGVyIGlmIGl0IGV4aXN0c1xuICAgICAgICBpZiAoZGF3bkVwb2NoID4gMCkge1xuICAgICAgICAgIHNldChcbiAgICAgICAgICAgIHJlcXVlc3RCb2R5LnF1ZXJ5LmJvb2wuZmlsdGVyLFxuICAgICAgICAgICAgYCR7ZmlsdGVyU2l6ZX0ucmFuZ2UuJHtGT1JFQ0FTVEVSX0RPQ19GSUVMRFMuRVhFQ1VUSU9OX0VORF9USU1FfS5ndGVgLFxuICAgICAgICAgICAgZGF3bkVwb2NoXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuICAgICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgICAgY29uc29sZS5sb2coJ3dyb25nIGRhdGUgcmFuZ2UgZmlsdGVyJywgZXJyb3IpO1xuICAgICAgfVxuXG4gICAgICAvLyDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIDilIBcbiAgICAgIC8vIElmIG1heEVudGl0aWVzID4gMCwgZmluZCB0b3AgTiBlbnRpdHlfaWRzLlxuICAgICAgLy8g4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSAXG4gICAgICBsZXQgcmVzdHJpY3RlZEVudGl0eUlkczogc3RyaW5nW10gPSBbXTtcblxuICAgICAgaWYgKG1heEVudGl0aWVzID4gMCkge1xuICAgICAgICBjb25zdCBlbnRpdHlMaXN0QXNPYmogPVxuICAgICAgICAgIGVudGl0eUxpc3QubGVuZ3RoID09PSAwID8ge30gOiBKU09OLnBhcnNlKGVudGl0eUxpc3QpO1xuXG4gICAgICAgIGNvbnN0IGVudGl0eUZpbHRlcnMgPSBpc0VtcHR5KGVudGl0eUxpc3RBc09iailcbiAgICAgICAgICA/IHt9XG4gICAgICAgICAgOiBidWlsZEVudGl0eUxpc3RRdWVyeShlbnRpdHlMaXN0QXNPYmopO1xuXG4gICAgICAgIC8vIE9ubHkgY2xvbmUgYW5kIG1vZGlmeSByZXF1ZXN0Qm9keSBpZiBlbnRpdHlGaWx0ZXJzIGV4aXN0cyBhbmQgaXMgbm90IGVtcHR5L251bGxcbiAgICAgICAgbGV0IHF1ZXJ5Rm9yQWdncmVnYXRpb247XG4gICAgICAgIGlmIChlbnRpdHlGaWx0ZXJzICYmIHR5cGVvZiBlbnRpdHlGaWx0ZXJzID09PSAnb2JqZWN0JyAmJiBPYmplY3Qua2V5cyhlbnRpdHlGaWx0ZXJzKS5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgLy8gQ3JlYXRlIGEgZGVlcCBjbG9uZSBvZiB0aGUgcmVxdWVzdCBib2R5XG4gICAgICAgICAgY29uc3QgY2xvbmVkUmVxdWVzdEJvZHkgPSBKU09OLnBhcnNlKEpTT04uc3RyaW5naWZ5KHJlcXVlc3RCb2R5KSk7XG5cbiAgICAgICAgICAvLyBBZGQgZW50aXR5IGZpbHRlcnMgdG8gdGhlIGZpbHRlciBhcnJheSBvZiB0aGUgY2xvbmVkIHJlcXVlc3QgYm9keVxuICAgICAgICAgIGlmICghY2xvbmVkUmVxdWVzdEJvZHkucXVlcnkpIHtcbiAgICAgICAgICAgIGNsb25lZFJlcXVlc3RCb2R5LnF1ZXJ5ID0geyBib29sOiB7IGZpbHRlcjogW10gfSB9O1xuICAgICAgICAgIH0gZWxzZSBpZiAoIWNsb25lZFJlcXVlc3RCb2R5LnF1ZXJ5LmJvb2wpIHtcbiAgICAgICAgICAgIGNsb25lZFJlcXVlc3RCb2R5LnF1ZXJ5LmJvb2wgPSB7IGZpbHRlcjogW10gfTtcbiAgICAgICAgICB9IGVsc2UgaWYgKCFjbG9uZWRSZXF1ZXN0Qm9keS5xdWVyeS5ib29sLmZpbHRlcikge1xuICAgICAgICAgICAgY2xvbmVkUmVxdWVzdEJvZHkucXVlcnkuYm9vbC5maWx0ZXIgPSBbXTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICAvLyBBZGQgdGhlIGVudGl0eSBmaWx0ZXIgb2JqZWN0IHRvIHRoZSBmaWx0ZXIgYXJyYXlcbiAgICAgICAgICBjbG9uZWRSZXF1ZXN0Qm9keS5xdWVyeS5ib29sLmZpbHRlci5wdXNoKGVudGl0eUZpbHRlcnMpO1xuXG4gICAgICAgICAgcXVlcnlGb3JBZ2dyZWdhdGlvbiA9IGNsb25lZFJlcXVlc3RCb2R5LnF1ZXJ5O1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIC8vIFVzZSB0aGUgb3JpZ2luYWwgcmVxdWVzdEJvZHkgaWYgbm8gZW50aXR5IGZpbHRlcnMgdG8gYWRkXG4gICAgICAgICAgcXVlcnlGb3JBZ2dyZWdhdGlvbiA9IHJlcXVlc3RCb2R5LnF1ZXJ5O1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gRXhhbXBsZSBhZ2dyZWdhdG9yUmVxdWVzdEJvZHk6XG4gICAgICAgIC8vIHtcbiAgICAgICAgLy8gICBcInNpemVcIjogMCxcbiAgICAgICAgLy8gICBcInF1ZXJ5XCI6IHtcbiAgICAgICAgLy8gICAgIFwiYm9vbFwiOiB7XG4gICAgICAgIC8vICAgICAgIFwiZmlsdGVyXCI6IFtcbiAgICAgICAgLy8gICAgICAgICB7XCJ0ZXJtXCI6IHtcInRhc2tfaWRcIjogXCJCc0xRYlpVQnhrd1FiMTRqOTNiRlwifX0sXG4gICAgICAgIC8vICAgICAgICAge1wicmFuZ2VcIjoge1wiZXhlY3V0aW9uX2VuZF90aW1lXCI6IHtcImd0ZVwiOiBcIjBcIn19fSxcbiAgICAgICAgLy8gICAgICAgICB7XG4gICAgICAgIC8vICAgICAgICAgICBcImJvb2xcIjoge1xuICAgICAgICAvLyAgICAgICAgICAgICBcInNob3VsZFwiOiBbXG4gICAgICAgIC8vICAgICAgICAgICAgICAge1xuICAgICAgICAvLyAgICAgICAgICAgICAgICAgXCJib29sXCI6IHtcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgXCJtdXN0XCI6IFtcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICBcIm5lc3RlZFwiOiB7XG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICAgIFwicGF0aFwiOiBcImVudGl0eVwiLFxuICAgICAgICAvLyAgICAgICAgICAgICAgICAgICAgICAgICBcInF1ZXJ5XCI6IHtcImJvb2xcIjoge1wibXVzdFwiOiBbe1widGVybVwiOiB7XCJlbnRpdHkubmFtZVwiOiBcInNlcnZpY2VcIn19LCB7XCJ0ZXJtXCI6IHtcImVudGl0eS52YWx1ZVwiOiBcImFwcF82XCJ9fV19fSxcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgICAgXCJpZ25vcmVfdW5tYXBwZWRcIjogZmFsc2UsXG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICAgIFwic2NvcmVfbW9kZVwiOiBcImF2Z1wiXG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICBcIm5lc3RlZFwiOiB7XG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICAgIFwicGF0aFwiOiBcImVudGl0eVwiLFxuICAgICAgICAvLyAgICAgICAgICAgICAgICAgICAgICAgICBcInF1ZXJ5XCI6IHtcImJvb2xcIjoge1wibXVzdFwiOiBbe1widGVybVwiOiB7XCJlbnRpdHkubmFtZVwiOiBcImhvc3RcIn19LCB7XCJ0ZXJtXCI6IHtcImVudGl0eS52YWx1ZVwiOiBcInNlcnZlcl8zXCJ9fV19fSxcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgICAgXCJpZ25vcmVfdW5tYXBwZWRcIjogZmFsc2UsXG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICAgIFwic2NvcmVfbW9kZVwiOiBcImF2Z1wiXG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAvLyAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgIC8vICAgICAgICAgICAgICAgICB9XG4gICAgICAgIC8vICAgICAgICAgICAgICAgfSxcbiAgICAgICAgLy8gICAgICAgICAgICAgICB7XG4gICAgICAgIC8vICAgICAgICAgICAgICAgICBcImJvb2xcIjoge1xuICAgICAgICAvLyAgICAgICAgICAgICAgICAgICBcIm11c3RcIjogW1xuICAgICAgICAvLyAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgIFwibmVzdGVkXCI6IHtcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgICAgXCJwYXRoXCI6IFwiZW50aXR5XCIsXG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICAgIFwicXVlcnlcIjoge1wiYm9vbFwiOiB7XCJtdXN0XCI6IFt7XCJ0ZXJtXCI6IHtcImVudGl0eS5uYW1lXCI6IFwic2VydmljZVwifX0sIHtcInRlcm1cIjoge1wiZW50aXR5LnZhbHVlXCI6IFwiYXBwXzZcIn19XX19LFxuICAgICAgICAvLyAgICAgICAgICAgICAgICAgICAgICAgICBcImlnbm9yZV91bm1hcHBlZFwiOiBmYWxzZSxcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgICAgXCJzY29yZV9tb2RlXCI6IFwiYXZnXCJcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAvLyAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgIFwibmVzdGVkXCI6IHtcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgICAgXCJwYXRoXCI6IFwiZW50aXR5XCIsXG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgICAgICAgIFwicXVlcnlcIjoge1wiYm9vbFwiOiB7XCJtdXN0XCI6IFt7XCJ0ZXJtXCI6IHtcImVudGl0eS5uYW1lXCI6IFwiaG9zdFwifX0sIHtcInRlcm1cIjoge1wiZW50aXR5LnZhbHVlXCI6IFwic2VydmVyXzFcIn19XX19LFxuICAgICAgICAvLyAgICAgICAgICAgICAgICAgICAgICAgICBcImlnbm9yZV91bm1hcHBlZFwiOiBmYWxzZSxcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgICAgXCJzY29yZV9tb2RlXCI6IFwiYXZnXCJcbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgLy8gICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgIC8vICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgLy8gICAgICAgICAgICAgICAgIH1cbiAgICAgICAgLy8gICAgICAgICAgICAgICB9XG4gICAgICAgIC8vICAgICAgICAgICAgIF0sXG4gICAgICAgIC8vICAgICAgICAgICAgIFwibWluaW11bV9zaG91bGRfbWF0Y2hcIjogMVxuICAgICAgICAvLyAgICAgICAgICAgfVxuICAgICAgICAvLyAgICAgICAgIH1cbiAgICAgICAgLy8gICAgICAgXVxuICAgICAgICAvLyAgICAgfVxuICAgICAgICAvLyAgIH0sXG4gICAgICAgIC8vICAgXCJhZ2dzXCI6IHtcbiAgICAgICAgLy8gICAgIFwidG9wX2VudGl0aWVzXCI6IHtcbiAgICAgICAgLy8gICAgICAgXCJ0ZXJtc1wiOiB7XG4gICAgICAgIC8vICAgICAgICAgXCJmaWVsZFwiOiBcImVudGl0eV9pZFwiLFxuICAgICAgICAvLyAgICAgICAgIFwic2l6ZVwiOiA1LFxuICAgICAgICAvLyAgICAgICAgIFwib3JkZXJcIjoge1wiX2NvdW50XCI6IFwiZGVzY1wifVxuICAgICAgICAvLyAgICAgICB9XG4gICAgICAgIC8vICAgICB9XG4gICAgICAgIC8vICAgfVxuICAgICAgICAvLyB9XG5cbiAgICAgICAgLy8gTm93IHVzZSB0aGUgYXBwcm9wcmlhdGUgcXVlcnkgaW4gYWdncmVnYXRvclJlcXVlc3RCb2R5XG4gICAgICAgIGNvbnN0IGFnZ3JlZ2F0b3JSZXF1ZXN0Qm9keSA9IHtcbiAgICAgICAgICBzaXplOiAwLFxuICAgICAgICAgIHF1ZXJ5OiBxdWVyeUZvckFnZ3JlZ2F0aW9uLFxuICAgICAgICAgIGFnZ3M6IHtcbiAgICAgICAgICAgIHRvcF9lbnRpdGllczoge1xuICAgICAgICAgICAgICB0ZXJtczoge1xuICAgICAgICAgICAgICAgIGZpZWxkOiAnZW50aXR5X2lkJyxcbiAgICAgICAgICAgICAgICBzaXplOiBtYXhFbnRpdGllcyxcbiAgICAgICAgICAgICAgICBvcmRlcjogeyBfY291bnQ6ICdkZXNjJyB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9O1xuXG4gICAgICAgIC8vIFdlJ2xsIGNhbGwgdGhlIHNhbWUgb3IgY3VzdG9tIHNlYXJjaCBtZXRob2Q6XG4gICAgICAgIGNvbnN0IGNhbGxXaXRoUmVxdWVzdCA9IGdldENsaWVudEJhc2VkT25EYXRhU291cmNlKFxuICAgICAgICAgIGNvbnRleHQsXG4gICAgICAgICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCxcbiAgICAgICAgICByZXF1ZXN0LFxuICAgICAgICAgIGRhdGFTb3VyY2VJZCxcbiAgICAgICAgICB0aGlzLmNsaWVudFxuICAgICAgICApO1xuXG4gICAgICAgIGNvbnN0IGFnZ1Jlc3BvbnNlID0gIXJlc3VsdEluZGV4XG4gICAgICAgICAgPyBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoJ2ZvcmVjYXN0LnNlYXJjaFJlc3VsdHMnLCB7XG4gICAgICAgICAgICBib2R5OiBhZ2dyZWdhdG9yUmVxdWVzdEJvZHksXG4gICAgICAgICAgfSlcbiAgICAgICAgICA6IGF3YWl0IGNhbGxXaXRoUmVxdWVzdChcbiAgICAgICAgICAgICdmb3JlY2FzdC5zZWFyY2hSZXN1bHRzRnJvbUN1c3RvbVJlc3VsdEluZGV4JyxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgcmVzdWx0SW5kZXg6IHJlc3VsdEluZGV4LFxuICAgICAgICAgICAgICBib2R5OiBhZ2dyZWdhdG9yUmVxdWVzdEJvZHksXG4gICAgICAgICAgICB9XG4gICAgICAgICAgKTtcblxuICAgICAgICAvLyBFeHRyYWN0IHRvcCBlbnRpdHlfaWRzXG4gICAgICAgIGNvbnN0IHRvcEVudGl0eUJ1Y2tldHMgPSBnZXQoYWdnUmVzcG9uc2UsICdhZ2dyZWdhdGlvbnMudG9wX2VudGl0aWVzLmJ1Y2tldHMnLCBbXSk7XG4gICAgICAgIHJlc3RyaWN0ZWRFbnRpdHlJZHMgPSB0b3BFbnRpdHlCdWNrZXRzLm1hcCgoYjogYW55KSA9PiBiLmtleSk7XG5cbiAgICAgICAgLy8gSWYgbm8gZW50aXRpZXMgbWF0Y2hlZCwgcmV0dXJuIGVtcHR5XG4gICAgICAgIGlmICghcmVzdHJpY3RlZEVudGl0eUlkcy5sZW5ndGgpIHtcbiAgICAgICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgICAgICBib2R5OiB7XG4gICAgICAgICAgICAgIG9rOiB0cnVlLFxuICAgICAgICAgICAgICByZXNwb25zZToge1xuICAgICAgICAgICAgICAgIHRvdGFsUmVzdWx0czogMCxcbiAgICAgICAgICAgICAgICByZXN1bHRzOiBbXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgLy8g4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSAXG4gICAgICAvLyBBZGQgYSB0ZXJtcyBmaWx0ZXIgdG8gcmVzdHJpY3QgZmluYWwgaGl0cyBpZiB3ZSBoYXZlIHRvcCBlbnRpdGllc1xuICAgICAgLy8g4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSA4pSAXG4gICAgICBpZiAocmVzdHJpY3RlZEVudGl0eUlkcy5sZW5ndGggPiAwKSB7XG4gICAgICAgIHJlcXVlc3RCb2R5LnF1ZXJ5LmJvb2wuZmlsdGVyLnB1c2goe1xuICAgICAgICAgIHRlcm1zOiB7IGVudGl0eV9pZDogcmVzdHJpY3RlZEVudGl0eUlkcyB9LFxuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgbGV0IHJlcXVlc3RQYXJhbXMgPSB7XG4gICAgICAgIHJlc3VsdEluZGV4OiByZXN1bHRJbmRleCxcbiAgICAgIH0gYXMge307XG5cbiAgICAgIGNvbnN0IGNhbGxXaXRoUmVxdWVzdCA9IGdldENsaWVudEJhc2VkT25EYXRhU291cmNlKFxuICAgICAgICBjb250ZXh0LFxuICAgICAgICB0aGlzLmRhdGFTb3VyY2VFbmFibGVkLFxuICAgICAgICByZXF1ZXN0LFxuICAgICAgICBkYXRhU291cmNlSWQsXG4gICAgICAgIHRoaXMuY2xpZW50XG4gICAgICApO1xuXG4gICAgICAvLyBBZGQgcGFnaW5hdGlvbiB3aXRoIHNlYXJjaF9hZnRlclxuICAgICAgbGV0IGFsbFJlc3VsdHMgPSBbXTtcbiAgICAgIGxldCBsYXN0U29ydCA9IG51bGw7XG4gICAgICBsZXQgaGFzTW9yZVJlc3VsdHMgPSB0cnVlO1xuICAgICAgbGV0IHRvdGFsSGl0cyA9IDA7XG5cbiAgICAgIC8vIENyZWF0ZSBhIGNvcHkgb2YgeW91ciBleGlzdGluZyByZXF1ZXN0Qm9keSB0byB1c2UgaW4gdGhlIHBhZ2luYXRpb24gbG9vcFxuICAgICAgY29uc3QgcGFnaW5hdGVkUmVxdWVzdEJvZHkgPSB7XG4gICAgICAgIC4uLnJlcXVlc3RCb2R5LFxuICAgICAgICBcInRyYWNrX3RvdGFsX2hpdHNcIjogdHJ1ZSAgLy8gQWRkIHRoaXMgdG8gZW5zdXJlIGFjY3VyYXRlIHRvdGFsIGNvdW50IGZvciBsYXJnZSByZXN1bHQgc2V0c1xuICAgICAgfTtcblxuICAgICAgLy8gQWRkIHNvcnQgaWYgbm90IGFscmVhZHkgcHJlc2VudFxuICAgICAgaWYgKCFwYWdpbmF0ZWRSZXF1ZXN0Qm9keS5zb3J0KSB7XG4gICAgICAgIHBhZ2luYXRlZFJlcXVlc3RCb2R5LnNvcnQgPSBbXG4gICAgICAgICAgeyBbc29ydEZpZWxkXTogc29ydERpcmVjdGlvbi50b0xvd2VyQ2FzZSgpIH0sXG4gICAgICAgICAgeyBcIl9pZFwiOiBcImFzY1wiIH0gIC8vIFNlY29uZGFyeSBzb3J0IGZvciB0aWVicmVha2VyXG4gICAgICAgIF07XG4gICAgICB9XG5cbiAgICAgIC8vIEV4ZWN1dGUgcGFnaW5hdGVkIHNlYXJjaFxuICAgICAgd2hpbGUgKGhhc01vcmVSZXN1bHRzKSB7XG4gICAgICAgIC8vIEFkZCBzZWFyY2hfYWZ0ZXIgZm9yIHN1YnNlcXVlbnQgcGFnZXNcbiAgICAgICAgaWYgKGxhc3RTb3J0KSB7XG4gICAgICAgICAgcGFnaW5hdGVkUmVxdWVzdEJvZHkuc2VhcmNoX2FmdGVyID0gbGFzdFNvcnQ7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBZb3VyIGV4aXN0aW5nIEFQSSBjYWxsLCBidXQgd2l0aCBvdXIgcGFnaW5hdGVkIHJlcXVlc3QgYm9keVxuICAgICAgICBjb25zdCByZXNwb25zZSA9ICFyZXN1bHRJbmRleFxuICAgICAgICAgID8gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KCdmb3JlY2FzdC5zZWFyY2hSZXN1bHRzJywge1xuICAgICAgICAgICAgYm9keTogcGFnaW5hdGVkUmVxdWVzdEJvZHksXG4gICAgICAgICAgfSlcbiAgICAgICAgICA6IGF3YWl0IGNhbGxXaXRoUmVxdWVzdCgnZm9yZWNhc3Quc2VhcmNoUmVzdWx0c0Zyb21DdXN0b21SZXN1bHRJbmRleCcsIHtcbiAgICAgICAgICAgIC4uLnJlcXVlc3RQYXJhbXMsXG4gICAgICAgICAgICBib2R5OiBwYWdpbmF0ZWRSZXF1ZXN0Qm9keSxcbiAgICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBoaXRzID0gZ2V0KHJlc3BvbnNlLCAnaGl0cy5oaXRzJywgW10pO1xuXG4gICAgICAgIC8vIFRyYWNrIHRvdGFsIGhpdHMgZnJvbSBmaXJzdCBwYWdlXG4gICAgICAgIGlmICghbGFzdFNvcnQpIHtcbiAgICAgICAgICB0b3RhbEhpdHMgPSBnZXQocmVzcG9uc2UsICdoaXRzLnRvdGFsLnZhbHVlJywgMCk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoaGl0cy5sZW5ndGggPT09IDAgfHwgaGl0cy5sZW5ndGggPCBzaXplKSB7XG4gICAgICAgICAgaGFzTW9yZVJlc3VsdHMgPSBmYWxzZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChoaXRzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAvLyBTYXZlIHNvcnQgdmFsdWVzIGZyb20gbGFzdCBoaXQgZm9yIG5leHQgaXRlcmF0aW9uXG4gICAgICAgICAgbGFzdFNvcnQgPSBoaXRzW2hpdHMubGVuZ3RoIC0gMV0uc29ydDtcblxuICAgICAgICAgIC8vIENvbGxlY3QgcmVzdWx0c1xuICAgICAgICAgIGFsbFJlc3VsdHMucHVzaCguLi5oaXRzKTtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICBjb25zdCBncm91cGVkUmVzdWx0cyA9IG5ldyBNYXAoKTtcbiAgICAgIGFsbFJlc3VsdHMuZm9yRWFjaCgocmVzdWx0KSA9PiB7XG4gICAgICAgIGNvbnN0IHNvdXJjZSA9IHJlc3VsdC5fc291cmNlO1xuICAgICAgICBjb25zdCBrZXkgPSBgJHtzb3VyY2UuZm9yZWNhc3Rlcl9pZH18JHtzb3VyY2UuZW50aXR5X2lkIHx8ICdkZWZhdWx0J318JHtzb3VyY2UuZGF0YV9lbmRfdGltZX1gO1xuXG4gICAgICAgIGlmICghZ3JvdXBlZFJlc3VsdHMuaGFzKGtleSkpIHtcbiAgICAgICAgICBncm91cGVkUmVzdWx0cy5zZXQoa2V5LCB7XG4gICAgICAgICAgICBmZWF0dXJlRGF0YTogbnVsbCxcbiAgICAgICAgICAgIGZvcmVjYXN0czogW11cbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChzb3VyY2UuZmVhdHVyZV9kYXRhKSB7XG4gICAgICAgICAgZ3JvdXBlZFJlc3VsdHMuZ2V0KGtleSkuZmVhdHVyZURhdGEgPSByZXN1bHQ7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgZ3JvdXBlZFJlc3VsdHMuZ2V0KGtleSkuZm9yZWNhc3RzLnB1c2gocmVzdWx0KTtcbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IGZvcmVjYXN0UmVzdWx0OiBGb3JlY2FzdFJlc3VsdFtdID0gW107XG5cbiAgICAgIC8vIFByb2Nlc3MgZWFjaCBncm91cFxuICAgICAgZ3JvdXBlZFJlc3VsdHMuZm9yRWFjaCgoeyBmZWF0dXJlRGF0YSwgZm9yZWNhc3RzIH0pID0+IHtcbiAgICAgICAgaWYgKCFmZWF0dXJlRGF0YSkgcmV0dXJuOyAvLyBTa2lwIGlmIG5vIGZlYXR1cmUgZGF0YSBmb3VuZFxuXG4gICAgICAgIC8vIENoZWNrIGlmIGFueSBmb3JlY2FzdCBoYXMgaG9yaXpvbl9pbmRleFxuICAgICAgICBjb25zdCBoYXNIb3Jpem9uSW5kZXggPSBmb3JlY2FzdHMuc29tZShmb3JlY2FzdCA9PiBmb3JlY2FzdC5fc291cmNlLmhvcml6b25faW5kZXggIT0gbnVsbCk7XG5cbiAgICAgICAgaWYgKGhhc0hvcml6b25JbmRleCkge1xuICAgICAgICAgIC8vIFNvcnQgZm9yZWNhc3RzIGJ5IGhvcml6b25faW5kZXggYW5kIGNvbWJpbmUgaW50byBhcnJheXNcbiAgICAgICAgICBjb25zdCBzb3J0ZWRGb3JlY2FzdHMgPSBvcmRlckJ5KGZvcmVjYXN0cywgWydfc291cmNlLmhvcml6b25faW5kZXgnXSwgWydhc2MnXSk7XG5cbiAgICAgICAgICBjb25zdCBmb3JlY2FzdFZhbHVlczogbnVtYmVyW10gPSBbXTtcbiAgICAgICAgICBjb25zdCBmb3JlY2FzdExvd2VyQm91bmRzOiBudW1iZXJbXSA9IFtdO1xuICAgICAgICAgIGNvbnN0IGZvcmVjYXN0VXBwZXJCb3VuZHM6IG51bWJlcltdID0gW107XG4gICAgICAgICAgY29uc3QgZm9yZWNhc3RTdGFydFRpbWVzOiBudW1iZXJbXSA9IFtdO1xuICAgICAgICAgIGNvbnN0IGZvcmVjYXN0RW5kVGltZXM6IG51bWJlcltdID0gW107XG5cbiAgICAgICAgICBzb3J0ZWRGb3JlY2FzdHMuZm9yRWFjaCgoZm9yZWNhc3Q6IGFueSkgPT4ge1xuICAgICAgICAgICAgY29uc3Qgc291cmNlID0gZm9yZWNhc3QuX3NvdXJjZTtcblxuICAgICAgICAgICAgZm9yZWNhc3RWYWx1ZXMucHVzaChcbiAgICAgICAgICAgICAgc291cmNlLmZvcmVjYXN0X3ZhbHVlICE9IG51bGwgJiYgc291cmNlLmZvcmVjYXN0X3ZhbHVlICE9PSAnTmFOJ1xuICAgICAgICAgICAgICAgID8gdG9GaXhlZE51bWJlckZvckZvcmVjYXN0KE51bWJlci5wYXJzZUZsb2F0KHNvdXJjZS5mb3JlY2FzdF92YWx1ZSkpXG4gICAgICAgICAgICAgICAgOiAwXG4gICAgICAgICAgICApO1xuXG4gICAgICAgICAgICBmb3JlY2FzdExvd2VyQm91bmRzLnB1c2goXG4gICAgICAgICAgICAgIHNvdXJjZS5mb3JlY2FzdF9sb3dlcl9ib3VuZCAhPSBudWxsICYmIHNvdXJjZS5mb3JlY2FzdF9sb3dlcl9ib3VuZCAhPT0gJ05hTidcbiAgICAgICAgICAgICAgICA/IHRvRml4ZWROdW1iZXJGb3JGb3JlY2FzdChOdW1iZXIucGFyc2VGbG9hdChzb3VyY2UuZm9yZWNhc3RfbG93ZXJfYm91bmQpKVxuICAgICAgICAgICAgICAgIDogMFxuICAgICAgICAgICAgKTtcblxuICAgICAgICAgICAgZm9yZWNhc3RVcHBlckJvdW5kcy5wdXNoKFxuICAgICAgICAgICAgICBzb3VyY2UuZm9yZWNhc3RfdXBwZXJfYm91bmQgIT0gbnVsbCAmJiBzb3VyY2UuZm9yZWNhc3RfdXBwZXJfYm91bmQgIT09ICdOYU4nXG4gICAgICAgICAgICAgICAgPyB0b0ZpeGVkTnVtYmVyRm9yRm9yZWNhc3QoTnVtYmVyLnBhcnNlRmxvYXQoc291cmNlLmZvcmVjYXN0X3VwcGVyX2JvdW5kKSlcbiAgICAgICAgICAgICAgICA6IDBcbiAgICAgICAgICAgICk7XG5cbiAgICAgICAgICAgIGZvcmVjYXN0U3RhcnRUaW1lcy5wdXNoKHNvdXJjZS5mb3JlY2FzdF9kYXRhX3N0YXJ0X3RpbWUpO1xuICAgICAgICAgICAgZm9yZWNhc3RFbmRUaW1lcy5wdXNoKHNvdXJjZS5mb3JlY2FzdF9kYXRhX2VuZF90aW1lKTtcbiAgICAgICAgICB9KTtcblxuICAgICAgICAgIGZvcmVjYXN0UmVzdWx0LnB1c2goe1xuICAgICAgICAgICAgc3RhcnRUaW1lOiBmZWF0dXJlRGF0YS5fc291cmNlLmRhdGFfc3RhcnRfdGltZSxcbiAgICAgICAgICAgIGVuZFRpbWU6IGZlYXR1cmVEYXRhLl9zb3VyY2UuZGF0YV9lbmRfdGltZSxcbiAgICAgICAgICAgIHBsb3RUaW1lOiBmZWF0dXJlRGF0YS5fc291cmNlLmRhdGFfZW5kX3RpbWUsXG4gICAgICAgICAgICBmb3JlY2FzdFZhbHVlOiBmb3JlY2FzdFZhbHVlcyxcbiAgICAgICAgICAgIGZvcmVjYXN0TG93ZXJCb3VuZDogZm9yZWNhc3RMb3dlckJvdW5kcyxcbiAgICAgICAgICAgIGZvcmVjYXN0VXBwZXJCb3VuZDogZm9yZWNhc3RVcHBlckJvdW5kcyxcbiAgICAgICAgICAgIGZvcmVjYXN0U3RhcnRUaW1lOiBmb3JlY2FzdFN0YXJ0VGltZXMsXG4gICAgICAgICAgICBmb3JlY2FzdEVuZFRpbWU6IGZvcmVjYXN0RW5kVGltZXMsXG4gICAgICAgICAgICAuLi4oZmVhdHVyZURhdGEuX3NvdXJjZS5lbnRpdHkgIT0gbnVsbCA/IHsgZW50aXR5OiBmZWF0dXJlRGF0YS5fc291cmNlLmVudGl0eSwgZW50aXR5SWQ6IGZlYXR1cmVEYXRhLl9zb3VyY2UuZW50aXR5X2lkIH0gOiB7fSksXG4gICAgICAgICAgICBmZWF0dXJlczogdGhpcy5nZXRGZWF0dXJlRGF0YShmZWF0dXJlRGF0YSlcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAvLyBEaXJlY3QgcHVzaCBmb3Igc2luZ2xlIGZvcmVjYXN0cyB3aXRob3V0IGhvcml6b25faW5kZXhcbiAgICAgICAgICBmb3JlY2FzdFJlc3VsdC5wdXNoKHtcbiAgICAgICAgICAgIHN0YXJ0VGltZTogZmVhdHVyZURhdGEuX3NvdXJjZS5kYXRhX3N0YXJ0X3RpbWUsXG4gICAgICAgICAgICBlbmRUaW1lOiBmZWF0dXJlRGF0YS5fc291cmNlLmRhdGFfZW5kX3RpbWUsXG4gICAgICAgICAgICBwbG90VGltZTogZmVhdHVyZURhdGEuX3NvdXJjZS5kYXRhX2VuZF90aW1lLFxuICAgICAgICAgICAgZm9yZWNhc3RWYWx1ZTogW10sXG4gICAgICAgICAgICBmb3JlY2FzdExvd2VyQm91bmQ6IFtdLFxuICAgICAgICAgICAgZm9yZWNhc3RVcHBlckJvdW5kOiBbXSxcbiAgICAgICAgICAgIGZvcmVjYXN0U3RhcnRUaW1lOiBbXSxcbiAgICAgICAgICAgIGZvcmVjYXN0RW5kVGltZTogW10sXG4gICAgICAgICAgICAuLi4oZmVhdHVyZURhdGEuX3NvdXJjZS5lbnRpdHkgIT0gbnVsbCA/IHsgZW50aXR5OiBmZWF0dXJlRGF0YS5fc291cmNlLmVudGl0eSwgZW50aXR5SWQ6IGZlYXR1cmVEYXRhLl9zb3VyY2UuZW50aXR5X2lkIH0gOiB7fSksXG4gICAgICAgICAgICBmZWF0dXJlczogdGhpcy5nZXRGZWF0dXJlRGF0YShmZWF0dXJlRGF0YSlcbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICAgIC8vIFNvcnQgZmluYWwgcmVzdWx0cyBieSBwbG90VGltZVxuICAgICAgY29uc3Qgc29ydGVkRm9yZWNhc3RSZXN1bHQgPSBvcmRlckJ5KGZvcmVjYXN0UmVzdWx0LCBbJ3Bsb3RUaW1lJ10sIFsnYXNjJ10pO1xuXG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZToge1xuICAgICAgICAgICAgdG90YWxSZXN1bHRzOiB0b3RhbEhpdHMsXG4gICAgICAgICAgICByZXN1bHRzOiBzb3J0ZWRGb3JlY2FzdFJlc3VsdCxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBjb25zb2xlLmxvZygnRm9yZWNhc3QgLSBVbmFibGUgdG8gZ2V0IHJlc3VsdHMnLCBlcnIpO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBnZXRFcnJvck1lc3NhZ2UoZXJyKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBnZXRUb3BGb3JlY2FzdFJlc3VsdHMgPSBhc3luYyAoXG4gICAgY29udGV4dDogUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlOiBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeVxuICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueT4+ID0+IHtcbiAgICB0cnkge1xuICAgICAgbGV0IHsgZm9yZWNhc3RlcklkLCBpc1J1bk9uY2UgfSA9IHJlcXVlc3QucGFyYW1zIGFzIHtcbiAgICAgICAgZm9yZWNhc3RlcklkOiBzdHJpbmc7XG4gICAgICAgIGlzUnVuT25jZTogYW55O1xuICAgICAgfTtcbiAgICAgIGNvbnN0IHsgZGF0YVNvdXJjZUlkID0gJycgfSA9IHJlcXVlc3QucGFyYW1zIGFzIHsgZGF0YVNvdXJjZUlkPzogc3RyaW5nIH07XG5cbiAgICAgIGlzUnVuT25jZSA9IEpTT04ucGFyc2UoaXNSdW5PbmNlKSBhcyBib29sZWFuO1xuICAgICAgY29uc3QgcmVxdWVzdFBhdGggPSAnZm9yZWNhc3QudG9wRm9yZWNhc3RSZXN1bHRzJztcblxuICAgICAgY29uc3QgY2FsbFdpdGhSZXF1ZXN0ID0gZ2V0Q2xpZW50QmFzZWRPbkRhdGFTb3VyY2UoXG4gICAgICAgIGNvbnRleHQsXG4gICAgICAgIHRoaXMuZGF0YVNvdXJjZUVuYWJsZWQsXG4gICAgICAgIHJlcXVlc3QsXG4gICAgICAgIGRhdGFTb3VyY2VJZCxcbiAgICAgICAgdGhpcy5jbGllbnRcbiAgICAgICk7XG5cbiAgICAgIC8vIERlZmluZSBwcm9wZXIgdHlwZXMgZm9yIE9wZW5TZWFyY2ggcXVlcnkgc3RydWN0dXJlc1xuICAgICAgdHlwZSBPcGVuU2VhcmNoUXVlcnkgPSBSZWNvcmQ8c3RyaW5nLCBhbnk+O1xuXG4gICAgICBpbnRlcmZhY2UgU3ViQWdncmVnYXRpb24ge1xuICAgICAgICBhZ2dyZWdhdGlvbl9xdWVyeToge1xuICAgICAgICAgIFtrZXk6IHN0cmluZ106IHtcbiAgICAgICAgICAgIFthZ2dyZWdhdGlvblR5cGU6IHN0cmluZ106IHtcbiAgICAgICAgICAgICAgZmllbGQ6IHN0cmluZztcbiAgICAgICAgICAgIH07XG4gICAgICAgICAgfTtcbiAgICAgICAgfTtcbiAgICAgICAgb3JkZXI6ICdBU0MnIHwgJ0RFU0MnO1xuICAgICAgfVxuXG4gICAgICAvLyBFeHRyYWN0IHRoZSBxdWVyeSBwYXJhbWV0ZXJzIGZyb20gdGhlIHJlcXVlc3QgYm9keSB3aXRoIGRlZmF1bHRzXG4gICAgICBjb25zdCB7XG4gICAgICAgIHNwbGl0X2J5ID0gJycsXG4gICAgICAgIGZpbHRlcl9ieSA9ICcnLFxuICAgICAgICBidWlsZF9pbl9xdWVyeSA9ICcnLFxuICAgICAgICBmb3JlY2FzdF9mcm9tID0gMCxcbiAgICAgICAgdGhyZXNob2xkID0gMCxcbiAgICAgICAgcmVsYXRpb25fdG9fdGhyZXNob2xkID0gJycsXG4gICAgICAgIGZpbHRlcl9xdWVyeSA9IHt9LFxuICAgICAgICBzdWJhZ2dyZWdhdGlvbnMgPSBbXVxuICAgICAgfSA9IChyZXF1ZXN0LmJvZHkgfHwge30pIGFzIHtcbiAgICAgICAgc3BsaXRfYnk6IHN0cmluZztcbiAgICAgICAgZmlsdGVyX2J5OiBzdHJpbmc7XG4gICAgICAgIGJ1aWxkX2luX3F1ZXJ5OiBzdHJpbmc7XG4gICAgICAgIGZvcmVjYXN0X2Zyb206IG51bWJlcjtcbiAgICAgICAgdGhyZXNob2xkOiBudW1iZXI7XG4gICAgICAgIHJlbGF0aW9uX3RvX3RocmVzaG9sZDogc3RyaW5nO1xuICAgICAgICBmaWx0ZXJfcXVlcnk6IE9wZW5TZWFyY2hRdWVyeTtcbiAgICAgICAgc3ViYWdncmVnYXRpb25zOiBTdWJBZ2dyZWdhdGlvbltdO1xuICAgICAgfTtcblxuICAgICAgLy8gQnVpbGQgcXVlcnkgb2JqZWN0IHdpdGggYXBwcm9wcmlhdGUgcGFyYW1ldGVyc1xuICAgICAgY29uc3QgcXVlcnlCb2R5OiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge307XG5cbiAgICAgIC8vIEFkZCBzcGxpdF9ieSBpZiBwcmVzZW50XG4gICAgICBpZiAoc3BsaXRfYnkpIHtcbiAgICAgICAgcXVlcnlCb2R5LnNwbGl0X2J5ID0gc3BsaXRfYnk7XG4gICAgICB9XG5cbiAgICAgIC8vIEFkZCBmaWx0ZXJfYnkgYW5kIHJlbGF0ZWQgcGFyYW1ldGVyc1xuICAgICAgaWYgKGZpbHRlcl9ieSkge1xuICAgICAgICBxdWVyeUJvZHkuZmlsdGVyX2J5ID0gZmlsdGVyX2J5O1xuXG4gICAgICAgIGlmIChmaWx0ZXJfYnkgPT09ICdCVUlMRF9JTl9RVUVSWScgJiYgYnVpbGRfaW5fcXVlcnkpIHtcbiAgICAgICAgICBxdWVyeUJvZHkuYnVpbGRfaW5fcXVlcnkgPSBidWlsZF9pbl9xdWVyeTtcblxuICAgICAgICAgIC8vIEFkZCB0aHJlc2hvbGQgcGFyYW1ldGVycyBpZiBidWlsZF9pbl9xdWVyeSBpcyBESVNUQU5DRV9UT19USFJFU0hPTERfVkFMVUVcbiAgICAgICAgICBpZiAoYnVpbGRfaW5fcXVlcnkgPT09ICdESVNUQU5DRV9UT19USFJFU0hPTERfVkFMVUUnKSB7XG4gICAgICAgICAgICBxdWVyeUJvZHkudGhyZXNob2xkID0gdGhyZXNob2xkO1xuICAgICAgICAgICAgcXVlcnlCb2R5LnJlbGF0aW9uX3RvX3RocmVzaG9sZCA9IHJlbGF0aW9uX3RvX3RocmVzaG9sZDtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSBpZiAoZmlsdGVyX2J5ID09PSAnQ1VTVE9NX1FVRVJZJykge1xuICAgICAgICAgIC8vIEFkZCBjdXN0b20gcXVlcnkgcGFyYW1ldGVycyAtIGNoZWNrIGlmIHRoZSBvYmplY3RzIGFyZSBub3QgZW1wdHlcbiAgICAgICAgICBpZiAoT2JqZWN0LmtleXMoZmlsdGVyX3F1ZXJ5KS5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICBxdWVyeUJvZHkuZmlsdGVyX3F1ZXJ5ID0gZmlsdGVyX3F1ZXJ5O1xuICAgICAgICAgIH1cbiAgICAgICAgICBpZiAoc3ViYWdncmVnYXRpb25zLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgIHF1ZXJ5Qm9keS5zdWJhZ2dyZWdhdGlvbnMgPSBzdWJhZ2dyZWdhdGlvbnM7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIC8vIEFkZCBmb3JlY2FzdF9mcm9tIHRpbWVzdGFtcCBpZiBwcmVzZW50XG4gICAgICBpZiAoZm9yZWNhc3RfZnJvbSkge1xuICAgICAgICBxdWVyeUJvZHkuZm9yZWNhc3RfZnJvbSA9IGZvcmVjYXN0X2Zyb207XG4gICAgICB9XG5cbiAgICAgIC8vIEFkZCBydW5fb25jZSB0byBib2R5IGlmIGlzUnVuT25jZSBpcyB0cnVlXG4gICAgICBjb25zdCByZXF1ZXN0Qm9keSA9IHtcbiAgICAgICAgLi4ucXVlcnlCb2R5LFxuICAgICAgICAuLi4oaXNSdW5PbmNlICYmIHsgcnVuX29uY2U6IHRydWUgfSksXG4gICAgICB9O1xuXG4gICAgICBjb25zdCByZXNwb25zZSA9IGF3YWl0IGNhbGxXaXRoUmVxdWVzdChyZXF1ZXN0UGF0aCwge1xuICAgICAgICBmb3JlY2FzdGVySWQ6IGZvcmVjYXN0ZXJJZCxcbiAgICAgICAgYm9keTogcmVxdWVzdEJvZHksXG4gICAgICB9KTtcblxuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IHRydWUsXG4gICAgICAgICAgcmVzcG9uc2U6IHJlc3BvbnNlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICBjb25zb2xlLmxvZygnRm9yZWNhc3QgLSBnZXRUb3BGb3JlY2FzdFJlc3VsdHMnLCBlcnIpO1xuICAgICAgcmV0dXJuIG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2Uub2soe1xuICAgICAgICBib2R5OiB7XG4gICAgICAgICAgb2s6IGZhbHNlLFxuICAgICAgICAgIGVycm9yOiBnZXRFcnJvck1lc3NhZ2UoZXJyKSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBtYXRjaEZvcmVjYXN0ZXIgPSBhc3luYyAoXG4gICAgY29udGV4dDogUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlOiBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeVxuICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueT4+ID0+IHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgeyBmb3JlY2FzdGVyTmFtZSB9ID0gcmVxdWVzdC5wYXJhbXMgYXMgeyBmb3JlY2FzdGVyTmFtZTogc3RyaW5nIH07XG4gICAgICBjb25zdCB7IGRhdGFTb3VyY2VJZCA9ICcnIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGRhdGFTb3VyY2VJZD86IHN0cmluZyB9O1xuXG4gICAgICBjb25zdCBjYWxsV2l0aFJlcXVlc3QgPSBnZXRDbGllbnRCYXNlZE9uRGF0YVNvdXJjZShcbiAgICAgICAgY29udGV4dCxcbiAgICAgICAgdGhpcy5kYXRhU291cmNlRW5hYmxlZCxcbiAgICAgICAgcmVxdWVzdCxcbiAgICAgICAgZGF0YVNvdXJjZUlkLFxuICAgICAgICB0aGlzLmNsaWVudFxuICAgICAgKTtcblxuICAgICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoXG4gICAgICAgICdmb3JlY2FzdC5tYXRjaEZvcmVjYXN0ZXInLCB7XG4gICAgICAgIGZvcmVjYXN0ZXJOYW1lLFxuICAgICAgfSk7XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZTogcmVzcG9uc2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUubG9nKCdGb3JlY2FzdCAtIG1hdGNoRm9yZWNhc3RlcicsIGVycik7XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHsgb2s6IGZhbHNlLCBlcnJvcjogZ2V0RXJyb3JNZXNzYWdlKGVycikgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBnZXRGb3JlY2FzdGVyQ291bnQgPSBhc3luYyAoXG4gICAgY29udGV4dDogUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlOiBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeVxuICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueT4+ID0+IHtcbiAgICB0cnkge1xuICAgICAgY29uc3QgeyBkYXRhU291cmNlSWQgPSAnJyB9ID0gcmVxdWVzdC5wYXJhbXMgYXMgeyBkYXRhU291cmNlSWQ/OiBzdHJpbmcgfTtcblxuICAgICAgY29uc3QgY2FsbFdpdGhSZXF1ZXN0ID0gZ2V0Q2xpZW50QmFzZWRPbkRhdGFTb3VyY2UoXG4gICAgICAgIGNvbnRleHQsXG4gICAgICAgIHRoaXMuZGF0YVNvdXJjZUVuYWJsZWQsXG4gICAgICAgIHJlcXVlc3QsXG4gICAgICAgIGRhdGFTb3VyY2VJZCxcbiAgICAgICAgdGhpcy5jbGllbnRcbiAgICAgICk7XG5cbiAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgY2FsbFdpdGhSZXF1ZXN0KFxuICAgICAgICAnZm9yZWNhc3QuZm9yZWNhc3RlckNvdW50Jyk7XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZTogcmVzcG9uc2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUubG9nKCdGb3JlY2FzdCAtIGdldEZvcmVjYXN0ZXJDb3VudCcsIGVycik7XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHsgb2s6IGZhbHNlLCBlcnJvcjogZ2V0RXJyb3JNZXNzYWdlKGVycikgfSxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcblxuICBnZXRGZWF0dXJlRGF0YSA9IChyYXdSZXN1bHQ6IGFueSkgPT4ge1xuICAgIGNvbnN0IGZlYXR1cmVSZXN1bHQ6IHsgW2tleTogc3RyaW5nXTogRmVhdHVyZVJlc3VsdCB9ID0ge307XG4gICAgcmF3UmVzdWx0Ll9zb3VyY2UuZmVhdHVyZV9kYXRhLmZvckVhY2goKGZlYXR1cmVEYXRhOiBhbnkpID0+IHtcbiAgICAgIGZlYXR1cmVSZXN1bHRbZmVhdHVyZURhdGEuZmVhdHVyZV9pZF0gPSB7XG4gICAgICAgIHN0YXJ0VGltZTogcmF3UmVzdWx0Ll9zb3VyY2UuZGF0YV9zdGFydF90aW1lLFxuICAgICAgICBlbmRUaW1lOiByYXdSZXN1bHQuX3NvdXJjZS5kYXRhX2VuZF90aW1lLFxuICAgICAgICBwbG90VGltZTogcmF3UmVzdWx0Ll9zb3VyY2UuZGF0YV9lbmRfdGltZSxcbiAgICAgICAgZGF0YTpcbiAgICAgICAgICBmZWF0dXJlRGF0YS5kYXRhICE9IG51bGwgJiYgZmVhdHVyZURhdGEuZGF0YSAhPT0gJ05hTidcbiAgICAgICAgICAgID8gdG9GaXhlZE51bWJlckZvckZvcmVjYXN0KE51bWJlci5wYXJzZUZsb2F0KGZlYXR1cmVEYXRhLmRhdGEpKVxuICAgICAgICAgICAgOiAwLFxuICAgICAgICBuYW1lOiBmZWF0dXJlRGF0YS5mZWF0dXJlX25hbWUsXG4gICAgICB9O1xuICAgIH0pO1xuICAgIHJldHVybiBmZWF0dXJlUmVzdWx0O1xuICB9O1xuXG4gIHNlYXJjaFJlc3VsdHMgPSBhc3luYyAoXG4gICAgY29udGV4dDogUmVxdWVzdEhhbmRsZXJDb250ZXh0LFxuICAgIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgICBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlOiBPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlRmFjdG9yeVxuICApOiBQcm9taXNlPElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlPGFueT4+ID0+IHtcbiAgICB0cnkge1xuICAgICAgdmFyIHsgcmVzdWx0SW5kZXggfSA9IHJlcXVlc3QucGFyYW1zIGFzIHtcbiAgICAgICAgcmVzdWx0SW5kZXg6IHN0cmluZztcbiAgICAgIH07XG4gICAgICBjb25zdCB7IGRhdGFTb3VyY2VJZCA9ICcnIH0gPSByZXF1ZXN0LnBhcmFtcyBhcyB7IGRhdGFTb3VyY2VJZD86IHN0cmluZyB9O1xuXG4gICAgICBpZiAoIXJlc3VsdEluZGV4IHx8ICFyZXN1bHRJbmRleC5zdGFydHNXaXRoKENVU1RPTV9GT1JFQ0FTVF9SRVNVTFRfSU5ERVhfUFJFRklYKSkge1xuICAgICAgICAvLyBTZXQgcmVzdWx0SW5kZXggYXMgJycgbWVhbnMgbm8gY3VzdG9tIHJlc3VsdCBpbmRleCBzcGVjaWZpZWQsIHdpbGwgb25seSBzZWFyY2ggZm9yZWNhc3QgcmVzdWx0IGZyb20gZGVmYXVsdCBpbmRleC5cbiAgICAgICAgcmVzdWx0SW5kZXggPSAnJztcbiAgICAgIH1cblxuICAgICAgbGV0IHJlcXVlc3RQYXJhbXMgPSB7XG4gICAgICAgIHJlc3VsdEluZGV4OiByZXN1bHRJbmRleCxcbiAgICAgIH0gYXMge307XG4gICAgICBjb25zdCByZXF1ZXN0Qm9keSA9IEpTT04uc3RyaW5naWZ5KHJlcXVlc3QuYm9keSk7XG5cbiAgICAgIGNvbnN0IGNhbGxXaXRoUmVxdWVzdCA9IGdldENsaWVudEJhc2VkT25EYXRhU291cmNlKFxuICAgICAgICBjb250ZXh0LFxuICAgICAgICB0aGlzLmRhdGFTb3VyY2VFbmFibGVkLFxuICAgICAgICByZXF1ZXN0LFxuICAgICAgICBkYXRhU291cmNlSWQsXG4gICAgICAgIHRoaXMuY2xpZW50XG4gICAgICApO1xuXG4gICAgICBjb25zdCByZXNwb25zZSA9ICFyZXN1bHRJbmRleFxuICAgICAgICA/IGF3YWl0IGNhbGxXaXRoUmVxdWVzdCgnZm9yZWNhc3Quc2VhcmNoUmVzdWx0cycsIHtcbiAgICAgICAgICBib2R5OiByZXF1ZXN0Qm9keSxcbiAgICAgICAgfSlcbiAgICAgICAgOiBhd2FpdCBjYWxsV2l0aFJlcXVlc3QoJ2ZvcmVjYXN0LnNlYXJjaFJlc3VsdHNGcm9tQ3VzdG9tUmVzdWx0SW5kZXgnLCB7XG4gICAgICAgICAgLi4ucmVxdWVzdFBhcmFtcyxcbiAgICAgICAgICBib2R5OiByZXF1ZXN0Qm9keSxcbiAgICAgICAgfSk7XG4gICAgICByZXR1cm4gb3BlbnNlYXJjaERhc2hib2FyZHNSZXNwb25zZS5vayh7XG4gICAgICAgIGJvZHk6IHtcbiAgICAgICAgICBvazogdHJ1ZSxcbiAgICAgICAgICByZXNwb25zZTogcmVzcG9uc2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIGNvbnNvbGUubG9nKCdGb3JlY2FzdCAtIFVuYWJsZSB0byBzZWFyY2ggZm9yZWNhc3QgcmVzdWx0JywgZXJyKTtcbiAgICAgIGlmIChpc0luZGV4Tm90Rm91bmRFcnJvcihlcnIpKSB7XG4gICAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgICBib2R5OiB7IG9rOiB0cnVlLCByZXNwb25zZTogeyB0b3RhbEZvcmVjYXN0ZXJzOiAwLCBmb3JlY2FzdGVyczogW10gfSB9LFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICAgIHJldHVybiBvcGVuc2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLm9rKHtcbiAgICAgICAgYm9keToge1xuICAgICAgICAgIG9rOiBmYWxzZSxcbiAgICAgICAgICBlcnJvcjogZ2V0RXJyb3JNZXNzYWdlKGVyciksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG4gIH07XG59XG4iXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFXQSxJQUFBQSxPQUFBLEdBQUFDLE9BQUE7QUFTQSxJQUFBQyxVQUFBLEdBQUFELE9BQUE7QUFNQSxJQUFBRSxRQUFBLEdBQUFGLE9BQUE7QUFJQSxJQUFBRyxnQkFBQSxHQUFBSCxPQUFBO0FBVWlDLFNBQUFJLGdCQUFBQyxDQUFBLEVBQUFDLENBQUEsRUFBQUMsQ0FBQSxZQUFBRCxDQUFBLEdBQUFFLGNBQUEsQ0FBQUYsQ0FBQSxNQUFBRCxDQUFBLEdBQUFJLE1BQUEsQ0FBQUMsY0FBQSxDQUFBTCxDQUFBLEVBQUFDLENBQUEsSUFBQUssS0FBQSxFQUFBSixDQUFBLEVBQUFLLFVBQUEsTUFBQUMsWUFBQSxNQUFBQyxRQUFBLFVBQUFULENBQUEsQ0FBQUMsQ0FBQSxJQUFBQyxDQUFBLEVBQUFGLENBQUE7QUFBQSxTQUFBRyxlQUFBRCxDQUFBLFFBQUFRLENBQUEsR0FBQUMsWUFBQSxDQUFBVCxDQUFBLHVDQUFBUSxDQUFBLEdBQUFBLENBQUEsR0FBQUEsQ0FBQTtBQUFBLFNBQUFDLGFBQUFULENBQUEsRUFBQUQsQ0FBQSwyQkFBQUMsQ0FBQSxLQUFBQSxDQUFBLFNBQUFBLENBQUEsTUFBQUYsQ0FBQSxHQUFBRSxDQUFBLENBQUFVLE1BQUEsQ0FBQUMsV0FBQSxrQkFBQWIsQ0FBQSxRQUFBVSxDQUFBLEdBQUFWLENBQUEsQ0FBQWMsSUFBQSxDQUFBWixDQUFBLEVBQUFELENBQUEsdUNBQUFTLENBQUEsU0FBQUEsQ0FBQSxZQUFBSyxTQUFBLHlFQUFBZCxDQUFBLEdBQUFlLE1BQUEsR0FBQUMsTUFBQSxFQUFBZixDQUFBLEtBeENqQztBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQWdETyxTQUFTZ0Isc0JBQXNCQSxDQUFDQyxTQUFpQixFQUFFQyxlQUFnQyxFQUFFO0VBQzFGO0VBQ0FELFNBQVMsQ0FBQ0UsSUFBSSxDQUFDLGNBQWMsRUFBRUQsZUFBZSxDQUFDRSxhQUFhLENBQUM7RUFDN0RILFNBQVMsQ0FBQ0UsSUFBSSxDQUFDLDZCQUE2QixFQUFFRCxlQUFlLENBQUNFLGFBQWEsQ0FBQzs7RUFFNUU7RUFDQUgsU0FBUyxDQUFDSSxHQUFHLENBQUMsNkJBQTZCLEVBQUVILGVBQWUsQ0FBQ0UsYUFBYSxDQUFDO0VBQzNFSCxTQUFTLENBQUNJLEdBQUcsQ0FDWCw0Q0FBNEMsRUFDNUNILGVBQWUsQ0FBQ0UsYUFDbEIsQ0FBQzs7RUFFRDtFQUNBSCxTQUFTLENBQUNFLElBQUksQ0FBQyxzQkFBc0IsRUFBRUQsZUFBZSxDQUFDSSxnQkFBZ0IsQ0FBQzs7RUFFeEU7QUFDRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztFQUVFO0VBQ0FMLFNBQVMsQ0FBQ0UsSUFBSSxDQUFDLHdDQUF3QyxFQUFFRCxlQUFlLENBQUNLLGFBQWEsQ0FBQzs7RUFFdkY7RUFDQU4sU0FBUyxDQUFDRSxJQUFJLENBQ1osdURBQXVELEVBQ3ZERCxlQUFlLENBQUNLLGFBQ2xCLENBQUM7O0VBRUQ7RUFDQU4sU0FBUyxDQUFDRSxJQUFJLENBQ1oscURBQXFELEVBQ3JERCxlQUFlLENBQUNLLGFBQ2xCLENBQUM7O0VBRUQ7RUFDQU4sU0FBUyxDQUFDRSxJQUFJLENBQ1osOEVBQThFLEVBQzlFRCxlQUFlLENBQUNLLGFBQ2xCLENBQUM7O0VBRUQ7RUFDQU4sU0FBUyxDQUFDTyxHQUFHLENBQUMsb0JBQW9CLEVBQUVOLGVBQWUsQ0FBQ08sY0FBYyxDQUFDO0VBQ25FUixTQUFTLENBQUNPLEdBQUcsQ0FBQyxtQ0FBbUMsRUFBRU4sZUFBZSxDQUFDTyxjQUFjLENBQUM7O0VBRWxGO0VBQ0FSLFNBQVMsQ0FBQ0UsSUFBSSxDQUFDLHVDQUF1QyxFQUFFRCxlQUFlLENBQUNRLGlCQUFpQixDQUFDO0VBQzFGVCxTQUFTLENBQUNFLElBQUksQ0FBQyxzREFBc0QsRUFBRUQsZUFBZSxDQUFDUSxpQkFBaUIsQ0FBQzs7RUFFekc7RUFDQVQsU0FBUyxDQUFDTyxHQUFHLENBQ1gscURBQXFELEVBQ3JETixlQUFlLENBQUNTLGtCQUNsQixDQUFDO0VBQ0RWLFNBQVMsQ0FBQ08sR0FBRyxDQUNYLG9FQUFvRSxFQUNwRU4sZUFBZSxDQUFDUyxrQkFDbEIsQ0FBQzs7RUFFRDtFQUNBVixTQUFTLENBQUNXLE1BQU0sQ0FBQyw2QkFBNkIsRUFBRVYsZUFBZSxDQUFDVyxnQkFBZ0IsQ0FBQztFQUNqRlosU0FBUyxDQUFDVyxNQUFNLENBQ2QsNENBQTRDLEVBQzVDVixlQUFlLENBQUNXLGdCQUNsQixDQUFDOztFQUVEO0VBQ0FaLFNBQVMsQ0FBQ0UsSUFBSSxDQUFDLG1DQUFtQyxFQUFFRCxlQUFlLENBQUNZLGVBQWUsQ0FBQztFQUNwRmIsU0FBUyxDQUFDRSxJQUFJLENBQ1osa0RBQWtELEVBQ2xERCxlQUFlLENBQUNZLGVBQ2xCLENBQUM7O0VBRUQ7RUFDQWIsU0FBUyxDQUFDRSxJQUFJLENBQ1osa0NBQWtDLEVBQ2xDRCxlQUFlLENBQUNhLGNBQ2xCLENBQUM7RUFDRGQsU0FBUyxDQUFDRSxJQUFJLENBQ1osaURBQWlELEVBQ2pERCxlQUFlLENBQUNhLGNBQ2xCLENBQUM7RUFFRGQsU0FBUyxDQUFDTyxHQUFHLENBQ1gsc0NBQXNDLEVBQ3RDTixlQUFlLENBQUNjLG9CQUNsQixDQUFDOztFQUVEO0VBQ0FmLFNBQVMsQ0FBQ08sR0FBRyxDQUFDLDZCQUE2QixFQUFFTixlQUFlLENBQUNlLGFBQWEsQ0FBQztFQUMzRWhCLFNBQVMsQ0FBQ08sR0FBRyxDQUNYLDRDQUE0QyxFQUM1Q04sZUFBZSxDQUFDZSxhQUNsQixDQUFDOztFQUVEO0VBQ0FoQixTQUFTLENBQUNPLEdBQUcsQ0FBQyxzQ0FBc0MsRUFBRU4sZUFBZSxDQUFDZ0IsZUFBZSxDQUFDO0VBQ3RGakIsU0FBUyxDQUFDTyxHQUFHLENBQUMscURBQXFELEVBQUVOLGVBQWUsQ0FBQ2dCLGVBQWUsQ0FBQzs7RUFFckc7RUFDQWpCLFNBQVMsQ0FBQ08sR0FBRyxDQUFDLHFCQUFxQixFQUFFTixlQUFlLENBQUNpQixrQkFBa0IsQ0FBQztFQUN4RWxCLFNBQVMsQ0FBQ08sR0FBRyxDQUFDLG9DQUFvQyxFQUFFTixlQUFlLENBQUNpQixrQkFBa0IsQ0FBQzs7RUFFdkY7RUFDQWxCLFNBQVMsQ0FBQ0UsSUFBSSxDQUNaLHVEQUF1RCxFQUN2REQsZUFBZSxDQUFDa0IscUJBQ2xCLENBQUM7RUFDRG5CLFNBQVMsQ0FBQ0UsSUFBSSxDQUNaLHNFQUFzRSxFQUN0RUQsZUFBZSxDQUFDa0IscUJBQ2xCLENBQUM7O0VBRUQ7RUFDQW5CLFNBQVMsQ0FBQ0UsSUFBSSxDQUNaLHlDQUF5QyxFQUN6Q0QsZUFBZSxDQUFDbUIsa0JBQ2xCLENBQUM7RUFDRHBCLFNBQVMsQ0FBQ0UsSUFBSSxDQUNaLHdEQUF3RCxFQUN4REQsZUFBZSxDQUFDbUIsa0JBQ2xCLENBQUM7O0VBRUQ7RUFDQXBCLFNBQVMsQ0FBQ0UsSUFBSSxDQUNaLHFDQUFxQyxFQUNyQ0QsZUFBZSxDQUFDb0IsaUJBQ2xCLENBQUM7RUFDRHJCLFNBQVMsQ0FBQ0UsSUFBSSxDQUNaLG9EQUFvRCxFQUNwREQsZUFBZSxDQUFDb0IsaUJBQ2xCLENBQUM7QUFDSDtBQUVlLE1BQU1DLGVBQWUsQ0FBQztFQUluQ0MsV0FBV0EsQ0FBQ0MsTUFBVyxFQUFFQyxpQkFBMEIsRUFBRTtJQUFBN0MsZUFBQTtJQUFBQSxlQUFBO0lBQUFBLGVBQUEsMkJBS2xDLE9BQ2pCOEMsT0FBOEIsRUFDOUJDLE9BQW9DLEVBQ3BDQyw0QkFBaUUsS0FDakI7TUFDaEQsSUFBSTtRQUNGLE1BQU07VUFBRUM7UUFBYSxDQUFDLEdBQUdGLE9BQU8sQ0FBQ0csTUFBa0M7UUFDbkUsTUFBTTtVQUFFQyxZQUFZLEdBQUc7UUFBRyxDQUFDLEdBQUdKLE9BQU8sQ0FBQ0csTUFBbUM7UUFDekU7UUFDQTtRQUNBO1FBQ0EsTUFBTUUsZUFBZSxHQUFHLElBQUFDLG1DQUEwQixFQUNoRFAsT0FBTyxFQUNQLElBQUksQ0FBQ0QsaUJBQWlCLEVBQ3RCRSxPQUFPLEVBQ1BJLFlBQVksRUFDWixJQUFJLENBQUNQLE1BQ1AsQ0FBQztRQUVELE1BQU1VLFFBQVEsR0FBRyxNQUFNRixlQUFlLENBQUMsMkJBQTJCLEVBQUU7VUFDbEVIO1FBQ0YsQ0FBQyxDQUFDO1FBRUYsT0FBT0QsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxJQUFJO1lBQ1JELFFBQVEsRUFBRUE7VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPRyxHQUFHLEVBQUU7UUFDWkMsT0FBTyxDQUFDQyxHQUFHLENBQUMsNkJBQTZCLEVBQUVGLEdBQUcsQ0FBQztRQUMvQyxPQUFPVCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLEtBQUs7WUFDVEssS0FBSyxFQUFFLElBQUFDLGdDQUFlLEVBQUNKLEdBQUc7VUFDNUI7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7SUFBQXpELGVBQUEsNEJBRW1CLE9BQ2xCOEMsT0FBOEIsRUFDOUJDLE9BQW9DLEVBQ3BDQyw0QkFBaUUsS0FDakI7TUFDaEQsSUFBSTtRQUNGLE1BQU07VUFBRUMsWUFBWSxHQUFHLEVBQUU7VUFBRUUsWUFBWSxHQUFHO1FBQUcsQ0FBQyxHQUFHSixPQUFPLENBQUNHLE1BQTBEO1FBRW5ILE1BQU1FLGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7UUFDRCxNQUFNVSxRQUFRLEdBQUcsTUFBTUYsZUFBZSxDQUNwQyw0QkFBNEIsRUFBRTtVQUM5Qkg7UUFDRixDQUFDLENBQUM7UUFDRixPQUFPRCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLElBQUk7WUFDUjtZQUNBRCxRQUFRLEVBQUVBO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT0csR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLDhCQUE4QixFQUFFRixHQUFHLENBQUM7UUFDaEQsT0FBT1QsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxLQUFLO1lBQ1RLLEtBQUssRUFBRSxJQUFBQyxnQ0FBZSxFQUFDSixHQUFHO1VBQzVCO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF6RCxlQUFBLHdCQUVlLE9BQ2Q4QyxPQUE4QixFQUM5QkMsT0FBb0MsRUFDcENDLDRCQUFpRSxLQUNqQjtNQUNoRCxJQUFJO1FBQ0YsTUFBTTtVQUFFQztRQUFhLENBQUMsR0FBR0YsT0FBTyxDQUFDRyxNQUFrQztRQUNuRSxNQUFNO1VBQUVDLFlBQVksR0FBRztRQUFHLENBQUMsR0FBR0osT0FBTyxDQUFDRyxNQUFtQzs7UUFFekU7UUFDQSxNQUFNWSxPQUFPLEdBQUdmLE9BQU8sQ0FBQ1MsSUFBSSxDQUFDTyxLQUFLO1FBQ2xDO1FBQ0EsTUFBTUMsYUFBYSxHQUFHakIsT0FBTyxDQUFDUyxJQUFJLENBQUNTLFdBQVc7UUFFOUMsTUFBTUMsV0FBVyxHQUFHQyxJQUFJLENBQUNDLFNBQVMsQ0FDaEMsSUFBQUMsK0NBQThCLEVBQUN0QixPQUFPLENBQUNTLElBQUksQ0FDN0MsQ0FBQztRQUVELElBQUlOLE1BQTJCLEdBQUc7VUFDaENELFlBQVksRUFBRUEsWUFBWTtVQUMxQmEsT0FBTyxFQUFFQSxPQUFPO1VBQ2hCRSxhQUFhLEVBQUVBLGFBQWE7VUFDNUJSLElBQUksRUFBRVU7UUFDUixDQUFDO1FBQ0QsSUFBSVosUUFBUTtRQUVaLE1BQU1GLGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7UUFFRCxJQUFJLElBQUEwQixnQkFBUSxFQUFDUixPQUFPLENBQUMsSUFBSSxJQUFBUSxnQkFBUSxFQUFDTixhQUFhLENBQUMsRUFBRTtVQUNoRFYsUUFBUSxHQUFHLE1BQU1GLGVBQWUsQ0FBQywyQkFBMkIsRUFBRUYsTUFBTSxDQUFDO1FBQ3ZFLENBQUMsTUFBTTtVQUNMSSxRQUFRLEdBQUcsTUFBTUYsZUFBZSxDQUFDLDJCQUEyQixFQUFFO1lBQzVESSxJQUFJLEVBQUVOLE1BQU0sQ0FBQ007VUFDZixDQUFDLENBQUM7UUFDSjtRQUNBLE1BQU1lLElBQUksR0FBRztVQUNYLEdBQUdqQixRQUFRLENBQUNrQixVQUFVO1VBQ3RCQyxFQUFFLEVBQUVuQixRQUFRLENBQUNvQixHQUFHO1VBQ2hCVCxXQUFXLEVBQUVYLFFBQVEsQ0FBQ3FCLGFBQWE7VUFDbkNaLEtBQUssRUFBRVQsUUFBUSxDQUFDc0I7UUFDbEIsQ0FBQztRQUNELE9BQU81Qiw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLElBQUk7WUFDUkQsUUFBUSxFQUFFLElBQUF1QiwrQ0FBOEIsRUFBQ04sSUFBSTtVQUMvQztRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPZCxHQUFHLEVBQUU7UUFDWkMsT0FBTyxDQUFDQyxHQUFHLENBQUMsMEJBQTBCLEVBQUVGLEdBQUcsQ0FBQztRQUM1QyxPQUFPVCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLEtBQUs7WUFDVEssS0FBSyxFQUFFLElBQUFDLGdDQUFlLEVBQUNKLEdBQUc7VUFDNUI7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7SUFBQXpELGVBQUEsNkJBRW9CLE9BQ25COEMsT0FBOEIsRUFDOUJDLE9BQW9DLEVBQ3BDQyw0QkFBaUUsS0FDakI7TUFDaEQsSUFBSTtRQUNGLElBQUk7VUFBRThCO1FBQWUsQ0FBQyxHQUFHL0IsT0FBTyxDQUFDRyxNQUVoQztRQUNELE1BQU07VUFBRUMsWUFBWSxHQUFHO1FBQUcsQ0FBQyxHQUFHSixPQUFPLENBQUNHLE1BQW1DO1FBRXpFLE1BQU1FLGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7UUFFRCxNQUFNc0IsV0FBVyxHQUFHQyxJQUFJLENBQUNDLFNBQVMsQ0FDaEMsSUFBQUMsK0NBQThCLEVBQUN0QixPQUFPLENBQUNTLElBQUksQ0FDN0MsQ0FBQztRQUNELE1BQU1GLFFBQVEsR0FBRyxNQUFNRixlQUFlLENBQ3BDLDZCQUE2QixFQUFFO1VBQy9CSSxJQUFJLEVBQUVVLFdBQVc7VUFDakJZLGNBQWMsRUFBRUE7UUFDbEIsQ0FBQyxDQUFDO1FBQ0YsT0FBTzlCLDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsSUFBSTtZQUNSRCxRQUFRLEVBQUVBO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT0csR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLCtCQUErQixFQUFFRixHQUFHLENBQUM7UUFDakQsT0FBT1QsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxLQUFLO1lBQ1RLLEtBQUssRUFBRSxJQUFBQyxnQ0FBZSxFQUFDSixHQUFHO1VBQzVCO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF6RCxlQUFBLDRCQUVtQixPQUNsQjhDLE9BQThCLEVBQzlCQyxPQUFvQyxFQUNwQ0MsNEJBQWlFLEtBQ2pCO01BQ2hELElBQUk7UUFDRixJQUFJO1VBQUUrQjtRQUFZLENBQUMsR0FBR2hDLE9BQU8sQ0FBQ0csTUFFN0I7UUFDRCxNQUFNO1VBQUVDLFlBQVksR0FBRztRQUFHLENBQUMsR0FBR0osT0FBTyxDQUFDRyxNQUFtQztRQUV6RSxNQUFNRSxlQUFlLEdBQUcsSUFBQUMsbUNBQTBCLEVBQ2hEUCxPQUFPLEVBQ1AsSUFBSSxDQUFDRCxpQkFBaUIsRUFDdEJFLE9BQU8sRUFDUEksWUFBWSxFQUNaLElBQUksQ0FBQ1AsTUFDUCxDQUFDO1FBRUQsTUFBTXNCLFdBQVcsR0FBR0MsSUFBSSxDQUFDQyxTQUFTLENBQ2hDLElBQUFDLCtDQUE4QixFQUFDdEIsT0FBTyxDQUFDUyxJQUFJLENBQzdDLENBQUM7UUFDRCxNQUFNRixRQUFRLEdBQUcsTUFBTUYsZUFBZSxDQUNwQyw0QkFBNEIsRUFBRTtVQUM5QkksSUFBSSxFQUFFVSxXQUFXO1VBQ2pCYSxXQUFXLEVBQUVBO1FBQ2YsQ0FBQyxDQUFDO1FBQ0YsT0FBTy9CLDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsSUFBSTtZQUNSRCxRQUFRLEVBQUVBO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT0csR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLDhCQUE4QixFQUFFRixHQUFHLENBQUM7UUFDaEQsT0FBT1QsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxLQUFLO1lBQ1RLLEtBQUssRUFBRSxJQUFBQyxnQ0FBZSxFQUFDSixHQUFHO1VBQzVCO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF6RCxlQUFBLHdCQUVlLE9BQ2Q4QyxPQUE4QixFQUM5QkMsT0FBb0MsRUFDcENDLDRCQUFpRSxLQUNqQjtNQUNoRCxJQUFJO1FBQ0YsTUFBTTtVQUFFQztRQUFhLENBQUMsR0FBR0YsT0FBTyxDQUFDRyxNQUFrQztRQUNuRSxNQUFNO1VBQUVDLFlBQVksR0FBRztRQUFHLENBQUMsR0FBR0osT0FBTyxDQUFDRyxNQUFtQztRQUV6RSxNQUFNRSxlQUFlLEdBQUcsSUFBQUMsbUNBQTBCLEVBQ2hEUCxPQUFPLEVBQ1AsSUFBSSxDQUFDRCxpQkFBaUIsRUFDdEJFLE9BQU8sRUFDUEksWUFBWSxFQUNaLElBQUksQ0FBQ1AsTUFDUCxDQUFDO1FBRUQsTUFBTW9DLGtCQUFrQixHQUFHLE1BQU01QixlQUFlLENBQUMsd0JBQXdCLEVBQUU7VUFDekVIO1FBQ0YsQ0FBQyxDQUFDOztRQUVGO1FBQ0EsTUFBTWdDLFlBQVksR0FBRztVQUNuQlIsRUFBRSxFQUFFTyxrQkFBa0IsQ0FBQ04sR0FBRztVQUMxQlQsV0FBVyxFQUFFZSxrQkFBa0IsQ0FBQ0wsYUFBYTtVQUM3Q1osS0FBSyxFQUFFaUIsa0JBQWtCLENBQUNKLE9BQU87VUFDakM7VUFDQSxHQUFHLElBQUFNLCtDQUE4QixFQUFDRixrQkFBa0IsQ0FBQ1IsVUFBVTtRQUNqRSxDQUFDOztRQUVEO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7O1FBRUE7UUFDQTtRQUNBOztRQUVBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7O1FBRUE7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTtRQUNBO1FBQ0E7UUFDQTs7UUFFQSxNQUFNVyxnQkFBZ0IsR0FBRyxJQUFBQyxtREFBa0MsRUFDekRKLGtCQUFrQixDQUFDSyxhQUFhLEVBQ2hDTCxrQkFBa0IsQ0FBQ00sYUFBYSxFQUNoQ04sa0JBQWtCLENBQUNPLGNBQ3JCLENBQUM7O1FBRUQ7UUFDQTtRQUNBLE1BQU1DLGFBQWEsR0FBRztVQUNwQixHQUFHUCxZQUFZO1VBQ2YsR0FBR0U7UUFDTCxDQUFDO1FBRUQsT0FBT25DLDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsSUFBSTtZQUNSRCxRQUFRLEVBQUVrQztVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxDQUFDLE9BQU8vQixHQUFHLEVBQUU7UUFDWjtRQUNBO1FBQ0E7UUFDQSxJQUNFQSxHQUFHLENBQUNnQyxVQUFVLEtBQUssR0FBRyxFQUN0QjtVQUNBLE9BQU96Qyw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1lBQ3JDQyxJQUFJLEVBQUU7Y0FBRUQsRUFBRSxFQUFFLElBQUk7Y0FBRUQsUUFBUSxFQUFFLENBQUM7WUFBRTtVQUNqQyxDQUFDLENBQUM7UUFDSjtRQUNBSSxPQUFPLENBQUNDLEdBQUcsQ0FBQyxxQ0FBcUMsRUFBRUYsR0FBRyxDQUFDO1FBQ3ZELE9BQU9ULDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsS0FBSztZQUNUSyxLQUFLLEVBQUUsSUFBQUMsZ0NBQWUsRUFBQ0osR0FBRztVQUM1QjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQUFBekQsZUFBQSwwQkFFaUIsT0FDaEI4QyxPQUE4QixFQUM5QkMsT0FBb0MsRUFDcENDLDRCQUFpRSxLQUNqQjtNQUNoRCxJQUFJO1FBQUEsSUFBQTBDLGFBQUEsRUFBQUMsY0FBQTtRQUNGLE1BQU07VUFBRTFDO1FBQWEsQ0FBQyxHQUFHRixPQUFPLENBQUNHLE1BQWtDO1FBQ25FLE1BQU07VUFBRUMsWUFBWSxHQUFHO1FBQUcsQ0FBQyxHQUFHSixPQUFPLENBQUNHLE1BQW1DO1FBQ3pFO1FBQ0EsTUFBTTBDLFNBQVMsSUFBQUYsYUFBQSxHQUFHM0MsT0FBTyxDQUFDUyxJQUFJLGNBQUFrQyxhQUFBLHVCQUFaQSxhQUFBLENBQWNFLFNBQVM7UUFDekM7UUFDQSxNQUFNQyxPQUFPLElBQUFGLGNBQUEsR0FBRzVDLE9BQU8sQ0FBQ1MsSUFBSSxjQUFBbUMsY0FBQSx1QkFBWkEsY0FBQSxDQUFjRSxPQUFPO1FBQ3JDLElBQUlDLGFBQWEsR0FBRztVQUFFN0MsWUFBWSxFQUFFQTtRQUFhLENBQU87UUFDeEQsSUFBSThDLFdBQVcsR0FBRywwQkFBMEI7UUFFNUMsTUFBTTNDLGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7UUFFRCxNQUFNVSxRQUFRLEdBQUcsTUFBTUYsZUFBZSxDQUFDMkMsV0FBVyxFQUFFRCxhQUFhLENBQUM7UUFFbEUsT0FBTzlDLDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsSUFBSTtZQUNSRCxRQUFRLEVBQUVBO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT0csR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLDRCQUE0QixFQUFFRixHQUFHLENBQUM7UUFDOUMsT0FBT1QsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxLQUFLO1lBQ1RLLEtBQUssRUFBRSxJQUFBQyxnQ0FBZSxFQUFDSixHQUFHO1VBQzVCO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF6RCxlQUFBLHlCQUVnQixPQUNmOEMsT0FBOEIsRUFDOUJDLE9BQW9DLEVBQ3BDQyw0QkFBaUUsS0FDakI7TUFDaEQsSUFBSTtRQUNGO1FBQ0E7UUFDQSxJQUFJO1VBQUVDO1FBQWEsQ0FBQyxHQUFHRixPQUFPLENBQUNHLE1BRTlCO1FBQ0Q7UUFDQTtRQUNBO1FBQ0EsTUFBTTtVQUFFQyxZQUFZLEdBQUc7UUFBRyxDQUFDLEdBQUdKLE9BQU8sQ0FBQ0csTUFBbUM7UUFFekUsTUFBTTZDLFdBQVcsR0FBRyx5QkFBeUI7UUFFN0MsTUFBTTNDLGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7UUFFRCxNQUFNVSxRQUFRLEdBQUcsTUFBTUYsZUFBZSxDQUFDMkMsV0FBVyxFQUFFO1VBQ2xEOUMsWUFBWSxFQUFFQTtRQUNoQixDQUFDLENBQUM7UUFFRixPQUFPRCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLElBQUk7WUFDUkQsUUFBUSxFQUFFQTtVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxDQUFDLE9BQU9HLEdBQUcsRUFBRTtRQUNaQyxPQUFPLENBQUNDLEdBQUcsQ0FBQywyQkFBMkIsRUFBRUYsR0FBRyxDQUFDO1FBQzdDLE9BQU9ULDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsS0FBSztZQUNUSyxLQUFLLEVBQUUsSUFBQUMsZ0NBQWUsRUFBQ0osR0FBRztVQUM1QjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQUFBekQsZUFBQSwrQkFFc0IsT0FDckI4QyxPQUE4QixFQUM5QkMsT0FBb0MsRUFDcENDLDRCQUFpRSxLQUNqQjtNQUNoRCxJQUFJO1FBQ0YsTUFBTTtVQUFFQztRQUFhLENBQUMsR0FBR0YsT0FBTyxDQUFDRyxNQUFrQztRQUNuRSxNQUFNSSxRQUFRLEdBQUcsTUFBTSxJQUFJLENBQUNWLE1BQU0sQ0FDL0JvRCxRQUFRLENBQUNqRCxPQUFPLENBQUMsQ0FDakJrRCxpQkFBaUIsQ0FBQyw0QkFBNEIsRUFBRTtVQUMvQ2hEO1FBQ0YsQ0FBQyxDQUFDO1FBQ0osT0FBT0QsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxJQUFJO1lBQ1JEO1VBQ0Y7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT0csR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLDhCQUE4QixFQUFFRixHQUFHLENBQUM7UUFDaEQsT0FBT1QsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxLQUFLO1lBQ1RLLEtBQUssRUFBRSxJQUFBQyxnQ0FBZSxFQUFDSixHQUFHO1VBQzVCO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF6RCxlQUFBLDJCQUVrQixPQUNqQjhDLE9BQThCLEVBQzlCQyxPQUFvQyxFQUNwQ0MsNEJBQWlFLEtBQ2pCO01BQ2hELElBQUk7UUFDRixNQUFNa0IsV0FBVyxHQUFHQyxJQUFJLENBQUNDLFNBQVMsQ0FBQ3JCLE9BQU8sQ0FBQ1MsSUFBSSxDQUFDO1FBQ2hELE1BQU1GLFFBQW9DLEdBQUcsTUFBTSxJQUFJLENBQUNWLE1BQU0sQ0FDM0RvRCxRQUFRLENBQUNqRCxPQUFPLENBQUMsQ0FDakJrRCxpQkFBaUIsQ0FBQywyQkFBMkIsRUFBRTtVQUFFekMsSUFBSSxFQUFFVTtRQUFZLENBQUMsQ0FBQztRQUN4RSxNQUFNZ0MsZ0JBQWdCLEdBQUcsSUFBQXZFLFdBQUcsRUFBQzJCLFFBQVEsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7UUFDN0QsTUFBTTZDLFdBQVcsR0FBRyxJQUFBeEUsV0FBRyxFQUFDMkIsUUFBUSxFQUFFLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQzhDLEdBQUcsQ0FBRTVCLFVBQWUsS0FBTTtVQUMzRSxHQUFHLElBQUFLLCtDQUE4QixFQUFDTCxVQUFVLENBQUM2QixPQUFPLENBQUM7VUFDckQ1QixFQUFFLEVBQUVELFVBQVUsQ0FBQ0UsR0FBRztVQUNsQlgsS0FBSyxFQUFFUyxVQUFVLENBQUNJLE9BQU87VUFDekJYLFdBQVcsRUFBRU8sVUFBVSxDQUFDRztRQUMxQixDQUFDLENBQUMsQ0FBQztRQUNILE9BQU8zQiw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLElBQUk7WUFDUkQsUUFBUSxFQUFFO2NBQ1I0QyxnQkFBZ0I7Y0FDaEJDO1lBQ0Y7VUFDRjtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPMUMsR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLHlDQUF5QyxFQUFFRixHQUFHLENBQUM7UUFDM0QsSUFBSSxJQUFBNkMscUNBQW9CLEVBQUM3QyxHQUFHLENBQUMsRUFBRTtVQUM3QixPQUFPVCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1lBQ3JDQyxJQUFJLEVBQUU7Y0FBRUQsRUFBRSxFQUFFLElBQUk7Y0FBRUQsUUFBUSxFQUFFO2dCQUFFNEMsZ0JBQWdCLEVBQUUsQ0FBQztnQkFBRUMsV0FBVyxFQUFFO2NBQUc7WUFBRTtVQUN2RSxDQUFDLENBQUM7UUFDSjtRQUNBLE9BQU9uRCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLEtBQUs7WUFDVEssS0FBSyxFQUFFLElBQUFDLGdDQUFlLEVBQUNKLEdBQUc7VUFDNUI7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7SUFBQXpELGVBQUEseUJBRWdCLE9BQ2Y4QyxPQUE4QixFQUM5QkMsT0FBb0MsRUFDcENDLDRCQUFpRSxLQUNqQjtNQUNoRCxJQUFJO1FBQ0YsTUFBTTtVQUNKRyxZQUFZLEdBQUc7UUFDakIsQ0FBQyxHQUFHSixPQUFPLENBQUNHLE1BQW1DO1FBRS9DLE1BQU1FLGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7UUFDRCxNQUFNVSxRQUFRLEdBQUcsTUFBTUYsZUFBZSxDQUFDLDJCQUEyQixFQUFFO1VBQ2xFSSxJQUFJLEVBQUU7WUFDSitDLElBQUksRUFBRUM7VUFDUjtRQUNGLENBQUMsQ0FBQztRQUNGLE1BQU1OLGdCQUFnQixHQUFHLElBQUF2RSxXQUFHLEVBQUMyQixRQUFRLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxDQUFDOztRQUU3RDtRQUNBLE1BQU1tRCxjQUFjLEdBQUcsSUFBQTlFLFdBQUcsRUFBQzJCLFFBQVEsRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUNvRCxNQUFNLENBQzFELENBQUNDLEdBQVEsRUFBRTNCLGtCQUF1QixNQUFNO1VBQ3RDLEdBQUcyQixHQUFHO1VBQ04sQ0FBQzNCLGtCQUFrQixDQUFDTixHQUFHLEdBQUc7WUFDeEJELEVBQUUsRUFBRU8sa0JBQWtCLENBQUNOLEdBQUc7WUFDMUJULFdBQVcsRUFBRWUsa0JBQWtCLENBQUNMLGFBQWE7WUFDN0NaLEtBQUssRUFBRWlCLGtCQUFrQixDQUFDSixPQUFPO1lBQ2pDLEdBQUcsSUFBQU0sK0NBQThCLEVBQUNGLGtCQUFrQixDQUFDcUIsT0FBTztVQUM5RDtRQUNGLENBQUMsQ0FBQyxFQUNGLENBQUMsQ0FDSCxDQUFDOztRQUVEO1FBQ0E7UUFDQTtRQUNBO1FBQ0EsSUFBSU8scUJBQXFCLEdBQUcsQ0FBQyxDQUFRO1FBQ3JDLElBQUlDLG9CQUFvQixHQUFHLENBQUMsQ0FBUTtRQUNwQyxJQUFJO1VBQ0ZELHFCQUFxQixHQUFHLE1BQU14RCxlQUFlLENBQUMsc0JBQXNCLEVBQUU7WUFDcEVJLElBQUksRUFBRSxJQUFBc0QsOENBQTZCLEVBQUMsSUFBSTtVQUMxQyxDQUFDLENBQUM7VUFDRkQsb0JBQW9CLEdBQUcsTUFBTXpELGVBQWUsQ0FBQyxzQkFBc0IsRUFBRTtZQUNuRUksSUFBSSxFQUFFLElBQUFzRCw4Q0FBNkIsRUFBQyxLQUFLO1VBQzNDLENBQUMsQ0FBQztRQUNKLENBQUMsQ0FBQyxPQUFPckQsR0FBRyxFQUFFO1VBQ1osSUFBSSxDQUFDLElBQUE2QyxxQ0FBb0IsRUFBQzdDLEdBQUcsQ0FBQyxFQUFFO1lBQzlCLE1BQU1BLEdBQUc7VUFDWDtRQUNGO1FBRUEsTUFBTXNELGFBQWEsR0FBRyxJQUFBcEYsV0FBRyxFQUN2QmlGLHFCQUFxQixFQUNyQixrQ0FBa0MsRUFDbEMsRUFDRixDQUFDLENBQUNGLE1BQU0sQ0FBQyxDQUFDQyxHQUFRLEVBQUVLLE1BQVcsS0FBSztVQUNsQyxPQUFPO1lBQ0wsR0FBR0wsR0FBRztZQUNOLENBQUNLLE1BQU0sQ0FBQ0MsR0FBRyxHQUFHO2NBQ1pDLFlBQVksRUFBRSxJQUFBdkYsV0FBRyxFQUFDcUYsTUFBTSxFQUFFLDBCQUEwQixFQUFFRyxTQUFTO1lBQ2pFO1VBQ0YsQ0FBQztRQUNILENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUVOLE1BQU1DLFlBQVksR0FBRyxJQUFBekYsV0FBRyxFQUN0QmtGLG9CQUFvQixFQUNwQixrQ0FBa0MsRUFDbEMsRUFDRixDQUFDLENBQUNILE1BQU0sQ0FBQyxDQUFDQyxHQUFRLEVBQUVLLE1BQVcsS0FBSztVQUNsQyxPQUFPO1lBQ0wsR0FBR0wsR0FBRztZQUNOLENBQUNLLE1BQU0sQ0FBQ0MsR0FBRyxHQUFHO2NBQ1pJLFdBQVcsRUFBRSxJQUFBMUYsV0FBRyxFQUFDcUYsTUFBTSxFQUFFLDBCQUEwQixFQUFFRyxTQUFTO1lBQ2hFO1VBQ0YsQ0FBQztRQUNILENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQzs7UUFFTjtRQUNBO1FBQ0E7UUFDQTtRQUNBLE1BQU1HLGdCQUFnQixHQUFHakgsTUFBTSxDQUFDa0gsTUFBTSxDQUFDZCxjQUFjLENBQUM7UUFDdERhLGdCQUFnQixDQUFDRSxPQUFPLENBQUVoRCxVQUFlLElBQUs7VUFDNUMsTUFBTTBDLFlBQVksR0FBRyxJQUFBdkYsV0FBRyxFQUN0Qm9GLGFBQWEsQ0FBQ3ZDLFVBQVUsQ0FBQ0MsRUFBRSxDQUFDLEVBQzVCLHNCQUNGLENBQUM7VUFDRCxNQUFNNEMsV0FBVyxHQUFHLElBQUExRixXQUFHLEVBQ3JCeUYsWUFBWSxDQUFDNUMsVUFBVSxDQUFDQyxFQUFFLENBQUMsRUFDM0IscUJBQ0YsQ0FBQztVQUVERCxVQUFVLENBQUNpRCxRQUFRLEdBQUcsSUFBQUMsaUNBQWdCLEVBQUNSLFlBQVksRUFBRUcsV0FBVyxDQUFDO1VBQ2pFN0MsVUFBVSxDQUFDbUQsc0JBQXNCLEdBQUcsSUFBQWhHLFdBQUcsRUFBQ3VGLFlBQVksRUFBRSxrQkFBa0IsQ0FBQztVQUN6RTFDLFVBQVUsQ0FBQ29ELHFCQUFxQixHQUFHLElBQUFqRyxXQUFHLEVBQUMwRixXQUFXLEVBQUUsa0JBQWtCLENBQUM7VUFDdkU3QyxVQUFVLENBQUNxRCxVQUFVLEdBQUcsSUFBQWxHLFdBQUcsRUFBQ3VGLFlBQVksRUFBRSxPQUFPLENBQUMsSUFBSSxJQUFBdkYsV0FBRyxFQUFDMEYsV0FBVyxFQUFFLE9BQU8sQ0FBQztRQUNqRixDQUFDLENBQUM7UUFFRixPQUFPckUsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxJQUFJO1lBQ1JELFFBQVEsRUFBRTtjQUNSNEMsZ0JBQWdCLEVBQUVBLGdCQUFnQjtjQUNsQzRCLGNBQWMsRUFBRVI7WUFDbEI7VUFDRjtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPN0QsR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLDJDQUEyQyxFQUFFRixHQUFHLENBQUM7UUFDN0QsSUFBSSxJQUFBNkMscUNBQW9CLEVBQUM3QyxHQUFHLENBQUMsRUFBRTtVQUM3QixPQUFPVCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1lBQ3JDQyxJQUFJLEVBQUU7Y0FBRUQsRUFBRSxFQUFFLElBQUk7Y0FBRUQsUUFBUSxFQUFFO2dCQUFFNEMsZ0JBQWdCLEVBQUUsQ0FBQztnQkFBRTRCLGNBQWMsRUFBRTtjQUFHO1lBQUU7VUFDMUUsQ0FBQyxDQUFDO1FBQ0o7UUFDQSxPQUFPOUUsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxLQUFLO1lBQ1RLLEtBQUssRUFBRSxJQUFBQyxnQ0FBZSxFQUFDSixHQUFHO1VBQzVCO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF6RCxlQUFBLDZCQUVvQixPQUNuQjhDLE9BQThCLEVBQzlCQyxPQUFvQyxFQUNwQ0MsNEJBQWlFLEtBQ2pCO01BQ2hELElBQUk7UUFBRXlCLEVBQUU7UUFBRXNELFNBQVM7UUFBRUM7TUFBYyxDQUFDLEdBQ2xDakYsT0FBTyxDQUFDRyxNQUlQO01BQ0gsTUFBTTtRQUFFQyxZQUFZLEdBQUc7TUFBRyxDQUFDLEdBQUdKLE9BQU8sQ0FBQ0csTUFBbUM7TUFFekUsSUFBSSxDQUFDOEUsV0FBVyxFQUFFO1FBQ2hCO1FBQ0FBLFdBQVcsR0FBRyxFQUFFO01BQ2xCLENBQUMsTUFBTSxJQUFJLENBQUNBLFdBQVcsQ0FBQ0MsVUFBVSxDQUFDQyw4Q0FBbUMsQ0FBQyxFQUFFO1FBQ3ZFO1FBQ0FGLFdBQVcsR0FBRyxFQUFFO01BQ2xCO01BRUFELFNBQVMsR0FBRzVELElBQUksQ0FBQ2dFLEtBQUssQ0FBQ0osU0FBUyxDQUFDOztNQUVqQztNQUNBLE1BQU1LLFVBQVUsR0FBR0wsU0FBUyxHQUFHO1FBQUVNLE9BQU8sRUFBRTVEO01BQUcsQ0FBQyxHQUFHO1FBQUU2RCxhQUFhLEVBQUU3RDtNQUFHLENBQUM7TUFFdEUsSUFBSTtRQUNGLE1BQU07VUFDSjhCLElBQUksR0FBRyxFQUFFO1VBQ1RnQyxhQUFhLEdBQUdDLHlCQUFjLENBQUNDLElBQUk7VUFDbkNDLFNBQVMsR0FBR0MsZ0NBQXFCLENBQUNDLGVBQWU7VUFDakRoRCxTQUFTLEdBQUcsQ0FBQztVQUNiQyxPQUFPLEdBQUcsQ0FBQztVQUNYZ0QsU0FBUyxHQUFHLEVBQUU7VUFDZEMsVUFBVSxHQUFHLEVBQUU7VUFDZkMsU0FBUyxHQUFHLENBQUM7VUFDYkMsV0FBVyxHQUFHO1FBQ2hCLENBQUMsR0FBR2pHLE9BQU8sQ0FBQ2tHLEtBVVg7O1FBRUQ7UUFDQSxNQUFNQyxZQUFZLEdBQUc7VUFDbkIsQ0FBQ1AsZ0NBQXFCLENBQUNDLGVBQWUsR0FBRztZQUN2QyxDQUFDRCxnQ0FBcUIsQ0FBQ0MsZUFBZSxHQUFHTDtVQUMzQyxDQUFDO1VBQ0QsQ0FBQ0ksZ0NBQXFCLENBQUNRLGFBQWEsR0FBRztZQUNyQyxDQUFDUixnQ0FBcUIsQ0FBQ1EsYUFBYSxHQUFHWjtVQUN6QztRQUNGLENBQThCO1FBQzlCLElBQUlhLElBQUksR0FBRyxDQUFDLENBQUM7UUFDYixNQUFNQyxTQUFTLEdBQUdILFlBQVksQ0FBQ1IsU0FBUyxDQUFDO1FBQ3pDLElBQUlXLFNBQVMsRUFBRTtVQUNiRCxJQUFJLEdBQUdDLFNBQVM7UUFDbEI7O1FBRUE7UUFDQSxNQUFNbkYsV0FBVyxHQUFHO1VBQ2xCa0YsSUFBSTtVQUNKN0MsSUFBSTtVQUNKMEMsS0FBSyxFQUFFO1lBQ0xLLElBQUksRUFBRTtjQUNKQyxNQUFNLEVBQUUsQ0FDTjtnQkFDRUMsSUFBSSxFQUFFcEI7Y0FDUixDQUFDO1lBRUw7VUFDRjtRQUNGLENBQUM7O1FBRUQ7UUFDQTtRQUNBLElBQUksQ0FBQ0wsU0FBUyxFQUFFO1VBQ2Q3RCxXQUFXLENBQUMrRSxLQUFLLENBQUNLLElBQUksR0FBRztZQUN2QixHQUFHcEYsV0FBVyxDQUFDK0UsS0FBSyxDQUFDSyxJQUFJO1lBQ3pCLEdBQUc7Y0FDREcsUUFBUSxFQUFFO2dCQUNSQyxNQUFNLEVBQUU7a0JBQ05DLEtBQUssRUFBRTtnQkFDVDtjQUNGO1lBQ0Y7VUFDRixDQUFDO1FBQ0g7UUFFQSxJQUFJO1VBQ0Y7VUFDQTtVQUNBLElBQUlDLFVBQVUsR0FBRzFGLFdBQVcsQ0FBQytFLEtBQUssQ0FBQ0ssSUFBSSxDQUFDQyxNQUFNLENBQUNNLE1BQU07VUFDckQsSUFBSWhCLFNBQVMsRUFBRTtZQUNiLENBQUNqRCxTQUFTLElBQUlDLE9BQU8sS0FDbkIsSUFBQWlFLFdBQUcsRUFDRDVGLFdBQVcsQ0FBQytFLEtBQUssQ0FBQ0ssSUFBSSxDQUFDQyxNQUFNLEVBQzVCLEdBQUVLLFVBQVcsVUFBU2YsU0FBVSxTQUFRLEVBQ3pDLGNBQ0YsQ0FBQztZQUVIakQsU0FBUyxJQUNQLElBQUFrRSxXQUFHLEVBQ0Q1RixXQUFXLENBQUMrRSxLQUFLLENBQUNLLElBQUksQ0FBQ0MsTUFBTSxFQUM1QixHQUFFSyxVQUFXLFVBQVNmLFNBQVUsTUFBSyxFQUN0Q2pELFNBQ0YsQ0FBQztZQUVIQyxPQUFPLElBQ0wsSUFBQWlFLFdBQUcsRUFDRDVGLFdBQVcsQ0FBQytFLEtBQUssQ0FBQ0ssSUFBSSxDQUFDQyxNQUFNLEVBQzVCLEdBQUVLLFVBQVcsVUFBU2YsU0FBVSxNQUFLLEVBQ3RDaEQsT0FDRixDQUFDO1VBQ0w7VUFFQStELFVBQVUsR0FBRzFGLFdBQVcsQ0FBQytFLEtBQUssQ0FBQ0ssSUFBSSxDQUFDQyxNQUFNLENBQUNNLE1BQU07O1VBRWpEO1VBQ0EsSUFBSWQsU0FBUyxHQUFHLENBQUMsRUFBRTtZQUNqQixJQUFBZSxXQUFHLEVBQ0Q1RixXQUFXLENBQUMrRSxLQUFLLENBQUNLLElBQUksQ0FBQ0MsTUFBTSxFQUM1QixHQUFFSyxVQUFXLFVBQVNqQixnQ0FBcUIsQ0FBQ29CLGtCQUFtQixNQUFLLEVBQ3JFaEIsU0FDRixDQUFDO1VBQ0g7UUFDRixDQUFDLENBQUMsT0FBT25GLEtBQUssRUFBRTtVQUNkRixPQUFPLENBQUNDLEdBQUcsQ0FBQyx5QkFBeUIsRUFBRUMsS0FBSyxDQUFDO1FBQy9DOztRQUVBO1FBQ0E7UUFDQTtRQUNBLElBQUlvRyxtQkFBNkIsR0FBRyxFQUFFO1FBRXRDLElBQUloQixXQUFXLEdBQUcsQ0FBQyxFQUFFO1VBQ25CLE1BQU1pQixlQUFlLEdBQ25CbkIsVUFBVSxDQUFDZSxNQUFNLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHMUYsSUFBSSxDQUFDZ0UsS0FBSyxDQUFDVyxVQUFVLENBQUM7VUFFdkQsTUFBTW9CLGFBQWEsR0FBRyxJQUFBQyxlQUFPLEVBQUNGLGVBQWUsQ0FBQyxHQUMxQyxDQUFDLENBQUMsR0FDRixJQUFBRyxxQ0FBb0IsRUFBQ0gsZUFBZSxDQUFDOztVQUV6QztVQUNBLElBQUlJLG1CQUFtQjtVQUN2QixJQUFJSCxhQUFhLElBQUksT0FBT0EsYUFBYSxLQUFLLFFBQVEsSUFBSTdKLE1BQU0sQ0FBQ2lLLElBQUksQ0FBQ0osYUFBYSxDQUFDLENBQUNMLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDL0Y7WUFDQSxNQUFNVSxpQkFBaUIsR0FBR3BHLElBQUksQ0FBQ2dFLEtBQUssQ0FBQ2hFLElBQUksQ0FBQ0MsU0FBUyxDQUFDRixXQUFXLENBQUMsQ0FBQzs7WUFFakU7WUFDQSxJQUFJLENBQUNxRyxpQkFBaUIsQ0FBQ3RCLEtBQUssRUFBRTtjQUM1QnNCLGlCQUFpQixDQUFDdEIsS0FBSyxHQUFHO2dCQUFFSyxJQUFJLEVBQUU7a0JBQUVDLE1BQU0sRUFBRTtnQkFBRztjQUFFLENBQUM7WUFDcEQsQ0FBQyxNQUFNLElBQUksQ0FBQ2dCLGlCQUFpQixDQUFDdEIsS0FBSyxDQUFDSyxJQUFJLEVBQUU7Y0FDeENpQixpQkFBaUIsQ0FBQ3RCLEtBQUssQ0FBQ0ssSUFBSSxHQUFHO2dCQUFFQyxNQUFNLEVBQUU7Y0FBRyxDQUFDO1lBQy9DLENBQUMsTUFBTSxJQUFJLENBQUNnQixpQkFBaUIsQ0FBQ3RCLEtBQUssQ0FBQ0ssSUFBSSxDQUFDQyxNQUFNLEVBQUU7Y0FDL0NnQixpQkFBaUIsQ0FBQ3RCLEtBQUssQ0FBQ0ssSUFBSSxDQUFDQyxNQUFNLEdBQUcsRUFBRTtZQUMxQzs7WUFFQTtZQUNBZ0IsaUJBQWlCLENBQUN0QixLQUFLLENBQUNLLElBQUksQ0FBQ0MsTUFBTSxDQUFDaUIsSUFBSSxDQUFDTixhQUFhLENBQUM7WUFFdkRHLG1CQUFtQixHQUFHRSxpQkFBaUIsQ0FBQ3RCLEtBQUs7VUFDL0MsQ0FBQyxNQUFNO1lBQ0w7WUFDQW9CLG1CQUFtQixHQUFHbkcsV0FBVyxDQUFDK0UsS0FBSztVQUN6Qzs7VUFFQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7VUFDQTtVQUNBO1VBQ0E7O1VBRUE7VUFDQSxNQUFNd0IscUJBQXFCLEdBQUc7WUFDNUJsRSxJQUFJLEVBQUUsQ0FBQztZQUNQMEMsS0FBSyxFQUFFb0IsbUJBQW1CO1lBQzFCSyxJQUFJLEVBQUU7Y0FDSkMsWUFBWSxFQUFFO2dCQUNaQyxLQUFLLEVBQUU7a0JBQ0xqQixLQUFLLEVBQUUsV0FBVztrQkFDbEJwRCxJQUFJLEVBQUV5QyxXQUFXO2tCQUNqQjZCLEtBQUssRUFBRTtvQkFBRUMsTUFBTSxFQUFFO2tCQUFPO2dCQUMxQjtjQUNGO1lBQ0Y7VUFDRixDQUFDOztVQUVEO1VBQ0EsTUFBTTFILGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7VUFFRCxNQUFNbUksV0FBVyxHQUFHLENBQUMvQyxXQUFXLEdBQzVCLE1BQU01RSxlQUFlLENBQUMsd0JBQXdCLEVBQUU7WUFDaERJLElBQUksRUFBRWlIO1VBQ1IsQ0FBQyxDQUFDLEdBQ0EsTUFBTXJILGVBQWUsQ0FDckIsNkNBQTZDLEVBQzdDO1lBQ0U0RSxXQUFXLEVBQUVBLFdBQVc7WUFDeEJ4RSxJQUFJLEVBQUVpSDtVQUNSLENBQ0YsQ0FBQzs7VUFFSDtVQUNBLE1BQU1PLGdCQUFnQixHQUFHLElBQUFySixXQUFHLEVBQUNvSixXQUFXLEVBQUUsbUNBQW1DLEVBQUUsRUFBRSxDQUFDO1VBQ2xGZixtQkFBbUIsR0FBR2dCLGdCQUFnQixDQUFDNUUsR0FBRyxDQUFFNkUsQ0FBTSxJQUFLQSxDQUFDLENBQUNoRSxHQUFHLENBQUM7O1VBRTdEO1VBQ0EsSUFBSSxDQUFDK0MsbUJBQW1CLENBQUNILE1BQU0sRUFBRTtZQUMvQixPQUFPN0csNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztjQUNyQ0MsSUFBSSxFQUFFO2dCQUNKRCxFQUFFLEVBQUUsSUFBSTtnQkFDUkQsUUFBUSxFQUFFO2tCQUNSNEgsWUFBWSxFQUFFLENBQUM7a0JBQ2ZDLE9BQU8sRUFBRTtnQkFDWDtjQUNGO1lBQ0YsQ0FBQyxDQUFDO1VBQ0o7UUFDRjs7UUFFQTtRQUNBO1FBQ0E7UUFDQSxJQUFJbkIsbUJBQW1CLENBQUNILE1BQU0sR0FBRyxDQUFDLEVBQUU7VUFDbEMzRixXQUFXLENBQUMrRSxLQUFLLENBQUNLLElBQUksQ0FBQ0MsTUFBTSxDQUFDaUIsSUFBSSxDQUFDO1lBQ2pDSSxLQUFLLEVBQUU7Y0FBRVEsU0FBUyxFQUFFcEI7WUFBb0I7VUFDMUMsQ0FBQyxDQUFDO1FBQ0o7UUFFQSxJQUFJbEUsYUFBYSxHQUFHO1VBQ2xCa0MsV0FBVyxFQUFFQTtRQUNmLENBQU87UUFFUCxNQUFNNUUsZUFBZSxHQUFHLElBQUFDLG1DQUEwQixFQUNoRFAsT0FBTyxFQUNQLElBQUksQ0FBQ0QsaUJBQWlCLEVBQ3RCRSxPQUFPLEVBQ1BJLFlBQVksRUFDWixJQUFJLENBQUNQLE1BQ1AsQ0FBQzs7UUFFRDtRQUNBLElBQUl5SSxVQUFVLEdBQUcsRUFBRTtRQUNuQixJQUFJQyxRQUFRLEdBQUcsSUFBSTtRQUNuQixJQUFJQyxjQUFjLEdBQUcsSUFBSTtRQUN6QixJQUFJQyxTQUFTLEdBQUcsQ0FBQzs7UUFFakI7UUFDQSxNQUFNQyxvQkFBb0IsR0FBRztVQUMzQixHQUFHdkgsV0FBVztVQUNkLGtCQUFrQixFQUFFLElBQUksQ0FBRTtRQUM1QixDQUFDOztRQUVEO1FBQ0EsSUFBSSxDQUFDdUgsb0JBQW9CLENBQUNyQyxJQUFJLEVBQUU7VUFDOUJxQyxvQkFBb0IsQ0FBQ3JDLElBQUksR0FBRyxDQUMxQjtZQUFFLENBQUNWLFNBQVMsR0FBR0gsYUFBYSxDQUFDbUQsV0FBVyxDQUFDO1VBQUUsQ0FBQyxFQUM1QztZQUFFLEtBQUssRUFBRTtVQUFNLENBQUMsQ0FBRTtVQUFBLENBQ25CO1FBQ0g7O1FBRUE7UUFDQSxPQUFPSCxjQUFjLEVBQUU7VUFDckI7VUFDQSxJQUFJRCxRQUFRLEVBQUU7WUFDWkcsb0JBQW9CLENBQUNFLFlBQVksR0FBR0wsUUFBUTtVQUM5Qzs7VUFFQTtVQUNBLE1BQU1oSSxRQUFRLEdBQUcsQ0FBQzBFLFdBQVcsR0FDekIsTUFBTTVFLGVBQWUsQ0FBQyx3QkFBd0IsRUFBRTtZQUNoREksSUFBSSxFQUFFaUk7VUFDUixDQUFDLENBQUMsR0FDQSxNQUFNckksZUFBZSxDQUFDLDZDQUE2QyxFQUFFO1lBQ3JFLEdBQUcwQyxhQUFhO1lBQ2hCdEMsSUFBSSxFQUFFaUk7VUFDUixDQUFDLENBQUM7VUFFSixNQUFNRyxJQUFJLEdBQUcsSUFBQWpLLFdBQUcsRUFBQzJCLFFBQVEsRUFBRSxXQUFXLEVBQUUsRUFBRSxDQUFDOztVQUUzQztVQUNBLElBQUksQ0FBQ2dJLFFBQVEsRUFBRTtZQUNiRSxTQUFTLEdBQUcsSUFBQTdKLFdBQUcsRUFBQzJCLFFBQVEsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7VUFDbEQ7VUFFQSxJQUFJc0ksSUFBSSxDQUFDL0IsTUFBTSxLQUFLLENBQUMsSUFBSStCLElBQUksQ0FBQy9CLE1BQU0sR0FBR3RELElBQUksRUFBRTtZQUMzQ2dGLGNBQWMsR0FBRyxLQUFLO1VBQ3hCO1VBRUEsSUFBSUssSUFBSSxDQUFDL0IsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNuQjtZQUNBeUIsUUFBUSxHQUFHTSxJQUFJLENBQUNBLElBQUksQ0FBQy9CLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQ1QsSUFBSTs7WUFFckM7WUFDQWlDLFVBQVUsQ0FBQ2IsSUFBSSxDQUFDLEdBQUdvQixJQUFJLENBQUM7VUFDMUI7UUFDRjtRQUVBLE1BQU1DLGNBQWMsR0FBRyxJQUFJQyxHQUFHLENBQUMsQ0FBQztRQUNoQ1QsVUFBVSxDQUFDN0QsT0FBTyxDQUFFdUUsTUFBTSxJQUFLO1VBQzdCLE1BQU1DLE1BQU0sR0FBR0QsTUFBTSxDQUFDMUYsT0FBTztVQUM3QixNQUFNWSxHQUFHLEdBQUksR0FBRStFLE1BQU0sQ0FBQzFELGFBQWMsSUFBRzBELE1BQU0sQ0FBQ1osU0FBUyxJQUFJLFNBQVUsSUFBR1ksTUFBTSxDQUFDQyxhQUFjLEVBQUM7VUFFOUYsSUFBSSxDQUFDSixjQUFjLENBQUNLLEdBQUcsQ0FBQ2pGLEdBQUcsQ0FBQyxFQUFFO1lBQzVCNEUsY0FBYyxDQUFDL0IsR0FBRyxDQUFDN0MsR0FBRyxFQUFFO2NBQ3RCa0YsV0FBVyxFQUFFLElBQUk7Y0FDakJDLFNBQVMsRUFBRTtZQUNiLENBQUMsQ0FBQztVQUNKO1VBRUEsSUFBSUosTUFBTSxDQUFDSyxZQUFZLEVBQUU7WUFDdkJSLGNBQWMsQ0FBQ2xLLEdBQUcsQ0FBQ3NGLEdBQUcsQ0FBQyxDQUFDa0YsV0FBVyxHQUFHSixNQUFNO1VBQzlDLENBQUMsTUFBTTtZQUNMRixjQUFjLENBQUNsSyxHQUFHLENBQUNzRixHQUFHLENBQUMsQ0FBQ21GLFNBQVMsQ0FBQzVCLElBQUksQ0FBQ3VCLE1BQU0sQ0FBQztVQUNoRDtRQUNGLENBQUMsQ0FBQztRQUVGLE1BQU1PLGNBQWdDLEdBQUcsRUFBRTs7UUFFM0M7UUFDQVQsY0FBYyxDQUFDckUsT0FBTyxDQUFDLENBQUM7VUFBRTJFLFdBQVc7VUFBRUM7UUFBVSxDQUFDLEtBQUs7VUFDckQsSUFBSSxDQUFDRCxXQUFXLEVBQUUsT0FBTyxDQUFDOztVQUUxQjtVQUNBLE1BQU1JLGVBQWUsR0FBR0gsU0FBUyxDQUFDSSxJQUFJLENBQUNDLFFBQVEsSUFBSUEsUUFBUSxDQUFDcEcsT0FBTyxDQUFDcUcsYUFBYSxJQUFJLElBQUksQ0FBQztVQUUxRixJQUFJSCxlQUFlLEVBQUU7WUFDbkI7WUFDQSxNQUFNSSxlQUFlLEdBQUcsSUFBQUMsZUFBTyxFQUFDUixTQUFTLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7WUFFOUUsTUFBTVMsY0FBd0IsR0FBRyxFQUFFO1lBQ25DLE1BQU1DLG1CQUE2QixHQUFHLEVBQUU7WUFDeEMsTUFBTUMsbUJBQTZCLEdBQUcsRUFBRTtZQUN4QyxNQUFNQyxrQkFBNEIsR0FBRyxFQUFFO1lBQ3ZDLE1BQU1DLGdCQUEwQixHQUFHLEVBQUU7WUFFckNOLGVBQWUsQ0FBQ25GLE9BQU8sQ0FBRWlGLFFBQWEsSUFBSztjQUN6QyxNQUFNVCxNQUFNLEdBQUdTLFFBQVEsQ0FBQ3BHLE9BQU87Y0FFL0J3RyxjQUFjLENBQUNyQyxJQUFJLENBQ2pCd0IsTUFBTSxDQUFDa0IsY0FBYyxJQUFJLElBQUksSUFBSWxCLE1BQU0sQ0FBQ2tCLGNBQWMsS0FBSyxLQUFLLEdBQzVELElBQUFDLGlDQUF3QixFQUFDak0sTUFBTSxDQUFDa00sVUFBVSxDQUFDcEIsTUFBTSxDQUFDa0IsY0FBYyxDQUFDLENBQUMsR0FDbEUsQ0FDTixDQUFDO2NBRURKLG1CQUFtQixDQUFDdEMsSUFBSSxDQUN0QndCLE1BQU0sQ0FBQ3FCLG9CQUFvQixJQUFJLElBQUksSUFBSXJCLE1BQU0sQ0FBQ3FCLG9CQUFvQixLQUFLLEtBQUssR0FDeEUsSUFBQUYsaUNBQXdCLEVBQUNqTSxNQUFNLENBQUNrTSxVQUFVLENBQUNwQixNQUFNLENBQUNxQixvQkFBb0IsQ0FBQyxDQUFDLEdBQ3hFLENBQ04sQ0FBQztjQUVETixtQkFBbUIsQ0FBQ3ZDLElBQUksQ0FDdEJ3QixNQUFNLENBQUNzQixvQkFBb0IsSUFBSSxJQUFJLElBQUl0QixNQUFNLENBQUNzQixvQkFBb0IsS0FBSyxLQUFLLEdBQ3hFLElBQUFILGlDQUF3QixFQUFDak0sTUFBTSxDQUFDa00sVUFBVSxDQUFDcEIsTUFBTSxDQUFDc0Isb0JBQW9CLENBQUMsQ0FBQyxHQUN4RSxDQUNOLENBQUM7Y0FFRE4sa0JBQWtCLENBQUN4QyxJQUFJLENBQUN3QixNQUFNLENBQUN1Qix3QkFBd0IsQ0FBQztjQUN4RE4sZ0JBQWdCLENBQUN6QyxJQUFJLENBQUN3QixNQUFNLENBQUN3QixzQkFBc0IsQ0FBQztZQUN0RCxDQUFDLENBQUM7WUFFRmxCLGNBQWMsQ0FBQzlCLElBQUksQ0FBQztjQUNsQjVFLFNBQVMsRUFBRXVHLFdBQVcsQ0FBQzlGLE9BQU8sQ0FBQ29ILGVBQWU7Y0FDOUM1SCxPQUFPLEVBQUVzRyxXQUFXLENBQUM5RixPQUFPLENBQUM0RixhQUFhO2NBQzFDeUIsUUFBUSxFQUFFdkIsV0FBVyxDQUFDOUYsT0FBTyxDQUFDNEYsYUFBYTtjQUMzQzBCLGFBQWEsRUFBRWQsY0FBYztjQUM3QmUsa0JBQWtCLEVBQUVkLG1CQUFtQjtjQUN2Q2Usa0JBQWtCLEVBQUVkLG1CQUFtQjtjQUN2Q2UsaUJBQWlCLEVBQUVkLGtCQUFrQjtjQUNyQ2UsZUFBZSxFQUFFZCxnQkFBZ0I7Y0FDakMsSUFBSWQsV0FBVyxDQUFDOUYsT0FBTyxDQUFDMkgsTUFBTSxJQUFJLElBQUksR0FBRztnQkFBRUEsTUFBTSxFQUFFN0IsV0FBVyxDQUFDOUYsT0FBTyxDQUFDMkgsTUFBTTtnQkFBRUMsUUFBUSxFQUFFOUIsV0FBVyxDQUFDOUYsT0FBTyxDQUFDK0U7Y0FBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7Y0FDOUg4QyxRQUFRLEVBQUUsSUFBSSxDQUFDQyxjQUFjLENBQUNoQyxXQUFXO1lBQzNDLENBQUMsQ0FBQztVQUNKLENBQUMsTUFBTTtZQUNMO1lBQ0FHLGNBQWMsQ0FBQzlCLElBQUksQ0FBQztjQUNsQjVFLFNBQVMsRUFBRXVHLFdBQVcsQ0FBQzlGLE9BQU8sQ0FBQ29ILGVBQWU7Y0FDOUM1SCxPQUFPLEVBQUVzRyxXQUFXLENBQUM5RixPQUFPLENBQUM0RixhQUFhO2NBQzFDeUIsUUFBUSxFQUFFdkIsV0FBVyxDQUFDOUYsT0FBTyxDQUFDNEYsYUFBYTtjQUMzQzBCLGFBQWEsRUFBRSxFQUFFO2NBQ2pCQyxrQkFBa0IsRUFBRSxFQUFFO2NBQ3RCQyxrQkFBa0IsRUFBRSxFQUFFO2NBQ3RCQyxpQkFBaUIsRUFBRSxFQUFFO2NBQ3JCQyxlQUFlLEVBQUUsRUFBRTtjQUNuQixJQUFJNUIsV0FBVyxDQUFDOUYsT0FBTyxDQUFDMkgsTUFBTSxJQUFJLElBQUksR0FBRztnQkFBRUEsTUFBTSxFQUFFN0IsV0FBVyxDQUFDOUYsT0FBTyxDQUFDMkgsTUFBTTtnQkFBRUMsUUFBUSxFQUFFOUIsV0FBVyxDQUFDOUYsT0FBTyxDQUFDK0U7Y0FBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7Y0FDOUg4QyxRQUFRLEVBQUUsSUFBSSxDQUFDQyxjQUFjLENBQUNoQyxXQUFXO1lBQzNDLENBQUMsQ0FBQztVQUNKO1FBQ0YsQ0FBQyxDQUFDOztRQUVGO1FBQ0EsTUFBTWlDLG9CQUFvQixHQUFHLElBQUF4QixlQUFPLEVBQUNOLGNBQWMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFM0UsT0FBT3RKLDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsSUFBSTtZQUNSRCxRQUFRLEVBQUU7Y0FDUjRILFlBQVksRUFBRU0sU0FBUztjQUN2QkwsT0FBTyxFQUFFaUQ7WUFDWDtVQUNGO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxDQUFDLE9BQU8zSyxHQUFHLEVBQUU7UUFDWkMsT0FBTyxDQUFDQyxHQUFHLENBQUMsa0NBQWtDLEVBQUVGLEdBQUcsQ0FBQztRQUNwRCxPQUFPVCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLEtBQUs7WUFDVEssS0FBSyxFQUFFLElBQUFDLGdDQUFlLEVBQUNKLEdBQUc7VUFDNUI7UUFDRixDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7SUFBQXpELGVBQUEsZ0NBRXVCLE9BQ3RCOEMsT0FBOEIsRUFDOUJDLE9BQW9DLEVBQ3BDQyw0QkFBaUUsS0FDakI7TUFDaEQsSUFBSTtRQUNGLElBQUk7VUFBRUMsWUFBWTtVQUFFOEU7UUFBVSxDQUFDLEdBQUdoRixPQUFPLENBQUNHLE1BR3pDO1FBQ0QsTUFBTTtVQUFFQyxZQUFZLEdBQUc7UUFBRyxDQUFDLEdBQUdKLE9BQU8sQ0FBQ0csTUFBbUM7UUFFekU2RSxTQUFTLEdBQUc1RCxJQUFJLENBQUNnRSxLQUFLLENBQUNKLFNBQVMsQ0FBWTtRQUM1QyxNQUFNaEMsV0FBVyxHQUFHLDZCQUE2QjtRQUVqRCxNQUFNM0MsZUFBZSxHQUFHLElBQUFDLG1DQUEwQixFQUNoRFAsT0FBTyxFQUNQLElBQUksQ0FBQ0QsaUJBQWlCLEVBQ3RCRSxPQUFPLEVBQ1BJLFlBQVksRUFDWixJQUFJLENBQUNQLE1BQ1AsQ0FBQzs7UUFFRDs7UUFjQTtRQUNBLE1BQU07VUFDSnlMLFFBQVEsR0FBRyxFQUFFO1VBQ2JDLFNBQVMsR0FBRyxFQUFFO1VBQ2RDLGNBQWMsR0FBRyxFQUFFO1VBQ25CQyxhQUFhLEdBQUcsQ0FBQztVQUNqQkMsU0FBUyxHQUFHLENBQUM7VUFDYkMscUJBQXFCLEdBQUcsRUFBRTtVQUMxQkMsWUFBWSxHQUFHLENBQUMsQ0FBQztVQUNqQkMsZUFBZSxHQUFHO1FBQ3BCLENBQUMsR0FBSTdMLE9BQU8sQ0FBQ1MsSUFBSSxJQUFJLENBQUMsQ0FTckI7O1FBRUQ7UUFDQSxNQUFNcUwsU0FBOEIsR0FBRyxDQUFDLENBQUM7O1FBRXpDO1FBQ0EsSUFBSVIsUUFBUSxFQUFFO1VBQ1pRLFNBQVMsQ0FBQ1IsUUFBUSxHQUFHQSxRQUFRO1FBQy9COztRQUVBO1FBQ0EsSUFBSUMsU0FBUyxFQUFFO1VBQ2JPLFNBQVMsQ0FBQ1AsU0FBUyxHQUFHQSxTQUFTO1VBRS9CLElBQUlBLFNBQVMsS0FBSyxnQkFBZ0IsSUFBSUMsY0FBYyxFQUFFO1lBQ3BETSxTQUFTLENBQUNOLGNBQWMsR0FBR0EsY0FBYzs7WUFFekM7WUFDQSxJQUFJQSxjQUFjLEtBQUssNkJBQTZCLEVBQUU7Y0FDcERNLFNBQVMsQ0FBQ0osU0FBUyxHQUFHQSxTQUFTO2NBQy9CSSxTQUFTLENBQUNILHFCQUFxQixHQUFHQSxxQkFBcUI7WUFDekQ7VUFDRixDQUFDLE1BQU0sSUFBSUosU0FBUyxLQUFLLGNBQWMsRUFBRTtZQUN2QztZQUNBLElBQUlqTyxNQUFNLENBQUNpSyxJQUFJLENBQUNxRSxZQUFZLENBQUMsQ0FBQzlFLE1BQU0sR0FBRyxDQUFDLEVBQUU7Y0FDeENnRixTQUFTLENBQUNGLFlBQVksR0FBR0EsWUFBWTtZQUN2QztZQUNBLElBQUlDLGVBQWUsQ0FBQy9FLE1BQU0sR0FBRyxDQUFDLEVBQUU7Y0FDOUJnRixTQUFTLENBQUNELGVBQWUsR0FBR0EsZUFBZTtZQUM3QztVQUNGO1FBQ0Y7O1FBRUE7UUFDQSxJQUFJSixhQUFhLEVBQUU7VUFDakJLLFNBQVMsQ0FBQ0wsYUFBYSxHQUFHQSxhQUFhO1FBQ3pDOztRQUVBO1FBQ0EsTUFBTXRLLFdBQVcsR0FBRztVQUNsQixHQUFHMkssU0FBUztVQUNaLElBQUk5RyxTQUFTLElBQUk7WUFBRStHLFFBQVEsRUFBRTtVQUFLLENBQUM7UUFDckMsQ0FBQztRQUVELE1BQU14TCxRQUFRLEdBQUcsTUFBTUYsZUFBZSxDQUFDMkMsV0FBVyxFQUFFO1VBQ2xEOUMsWUFBWSxFQUFFQSxZQUFZO1VBQzFCTyxJQUFJLEVBQUVVO1FBQ1IsQ0FBQyxDQUFDO1FBRUYsT0FBT2xCLDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsSUFBSTtZQUNSRCxRQUFRLEVBQUVBO1VBQ1o7UUFDRixDQUFDLENBQUM7TUFDSixDQUFDLENBQUMsT0FBT0csR0FBRyxFQUFFO1FBQ1pDLE9BQU8sQ0FBQ0MsR0FBRyxDQUFDLGtDQUFrQyxFQUFFRixHQUFHLENBQUM7UUFDcEQsT0FBT1QsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxLQUFLO1lBQ1RLLEtBQUssRUFBRSxJQUFBQyxnQ0FBZSxFQUFDSixHQUFHO1VBQzVCO1FBQ0YsQ0FBQyxDQUFDO01BQ0o7SUFDRixDQUFDO0lBQUF6RCxlQUFBLDBCQUVpQixPQUNoQjhDLE9BQThCLEVBQzlCQyxPQUFvQyxFQUNwQ0MsNEJBQWlFLEtBQ2pCO01BQ2hELElBQUk7UUFDRixNQUFNO1VBQUUrTDtRQUFlLENBQUMsR0FBR2hNLE9BQU8sQ0FBQ0csTUFBb0M7UUFDdkUsTUFBTTtVQUFFQyxZQUFZLEdBQUc7UUFBRyxDQUFDLEdBQUdKLE9BQU8sQ0FBQ0csTUFBbUM7UUFFekUsTUFBTUUsZUFBZSxHQUFHLElBQUFDLG1DQUEwQixFQUNoRFAsT0FBTyxFQUNQLElBQUksQ0FBQ0QsaUJBQWlCLEVBQ3RCRSxPQUFPLEVBQ1BJLFlBQVksRUFDWixJQUFJLENBQUNQLE1BQ1AsQ0FBQztRQUVELE1BQU1VLFFBQVEsR0FBRyxNQUFNRixlQUFlLENBQ3BDLDBCQUEwQixFQUFFO1VBQzVCMkw7UUFDRixDQUFDLENBQUM7UUFDRixPQUFPL0wsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxJQUFJO1lBQ1JELFFBQVEsRUFBRUE7VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPRyxHQUFHLEVBQUU7UUFDWkMsT0FBTyxDQUFDQyxHQUFHLENBQUMsNEJBQTRCLEVBQUVGLEdBQUcsQ0FBQztRQUM5QyxPQUFPVCw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFBRUQsRUFBRSxFQUFFLEtBQUs7WUFBRUssS0FBSyxFQUFFLElBQUFDLGdDQUFlLEVBQUNKLEdBQUc7VUFBRTtRQUNqRCxDQUFDLENBQUM7TUFDSjtJQUNGLENBQUM7SUFBQXpELGVBQUEsNkJBRW9CLE9BQ25COEMsT0FBOEIsRUFDOUJDLE9BQW9DLEVBQ3BDQyw0QkFBaUUsS0FDakI7TUFDaEQsSUFBSTtRQUNGLE1BQU07VUFBRUcsWUFBWSxHQUFHO1FBQUcsQ0FBQyxHQUFHSixPQUFPLENBQUNHLE1BQW1DO1FBRXpFLE1BQU1FLGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7UUFFRCxNQUFNVSxRQUFRLEdBQUcsTUFBTUYsZUFBZSxDQUNwQywwQkFBMEIsQ0FBQztRQUM3QixPQUFPSiw0QkFBNEIsQ0FBQ08sRUFBRSxDQUFDO1VBQ3JDQyxJQUFJLEVBQUU7WUFDSkQsRUFBRSxFQUFFLElBQUk7WUFDUkQsUUFBUSxFQUFFQTtVQUNaO1FBQ0YsQ0FBQyxDQUFDO01BQ0osQ0FBQyxDQUFDLE9BQU9HLEdBQUcsRUFBRTtRQUNaQyxPQUFPLENBQUNDLEdBQUcsQ0FBQywrQkFBK0IsRUFBRUYsR0FBRyxDQUFDO1FBQ2pELE9BQU9ULDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUFFRCxFQUFFLEVBQUUsS0FBSztZQUFFSyxLQUFLLEVBQUUsSUFBQUMsZ0NBQWUsRUFBQ0osR0FBRztVQUFFO1FBQ2pELENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQUFBekQsZUFBQSx5QkFFaUJnUCxTQUFjLElBQUs7TUFDbkMsTUFBTUMsYUFBK0MsR0FBRyxDQUFDLENBQUM7TUFDMURELFNBQVMsQ0FBQzNJLE9BQU8sQ0FBQ2dHLFlBQVksQ0FBQzdFLE9BQU8sQ0FBRTJFLFdBQWdCLElBQUs7UUFDM0Q4QyxhQUFhLENBQUM5QyxXQUFXLENBQUMrQyxVQUFVLENBQUMsR0FBRztVQUN0Q3RKLFNBQVMsRUFBRW9KLFNBQVMsQ0FBQzNJLE9BQU8sQ0FBQ29ILGVBQWU7VUFDNUM1SCxPQUFPLEVBQUVtSixTQUFTLENBQUMzSSxPQUFPLENBQUM0RixhQUFhO1VBQ3hDeUIsUUFBUSxFQUFFc0IsU0FBUyxDQUFDM0ksT0FBTyxDQUFDNEYsYUFBYTtVQUN6Q2tELElBQUksRUFDRmhELFdBQVcsQ0FBQ2dELElBQUksSUFBSSxJQUFJLElBQUloRCxXQUFXLENBQUNnRCxJQUFJLEtBQUssS0FBSyxHQUNsRCxJQUFBaEMsaUNBQXdCLEVBQUNqTSxNQUFNLENBQUNrTSxVQUFVLENBQUNqQixXQUFXLENBQUNnRCxJQUFJLENBQUMsQ0FBQyxHQUM3RCxDQUFDO1VBQ1BDLElBQUksRUFBRWpELFdBQVcsQ0FBQ2tEO1FBQ3BCLENBQUM7TUFDSCxDQUFDLENBQUM7TUFDRixPQUFPSixhQUFhO0lBQ3RCLENBQUM7SUFBQWpQLGVBQUEsd0JBRWUsT0FDZDhDLE9BQThCLEVBQzlCQyxPQUFvQyxFQUNwQ0MsNEJBQWlFLEtBQ2pCO01BQ2hELElBQUk7UUFDRixJQUFJO1VBQUVnRjtRQUFZLENBQUMsR0FBR2pGLE9BQU8sQ0FBQ0csTUFFN0I7UUFDRCxNQUFNO1VBQUVDLFlBQVksR0FBRztRQUFHLENBQUMsR0FBR0osT0FBTyxDQUFDRyxNQUFtQztRQUV6RSxJQUFJLENBQUM4RSxXQUFXLElBQUksQ0FBQ0EsV0FBVyxDQUFDQyxVQUFVLENBQUNDLDhDQUFtQyxDQUFDLEVBQUU7VUFDaEY7VUFDQUYsV0FBVyxHQUFHLEVBQUU7UUFDbEI7UUFFQSxJQUFJbEMsYUFBYSxHQUFHO1VBQ2xCa0MsV0FBVyxFQUFFQTtRQUNmLENBQU87UUFDUCxNQUFNOUQsV0FBVyxHQUFHQyxJQUFJLENBQUNDLFNBQVMsQ0FBQ3JCLE9BQU8sQ0FBQ1MsSUFBSSxDQUFDO1FBRWhELE1BQU1KLGVBQWUsR0FBRyxJQUFBQyxtQ0FBMEIsRUFDaERQLE9BQU8sRUFDUCxJQUFJLENBQUNELGlCQUFpQixFQUN0QkUsT0FBTyxFQUNQSSxZQUFZLEVBQ1osSUFBSSxDQUFDUCxNQUNQLENBQUM7UUFFRCxNQUFNVSxRQUFRLEdBQUcsQ0FBQzBFLFdBQVcsR0FDekIsTUFBTTVFLGVBQWUsQ0FBQyx3QkFBd0IsRUFBRTtVQUNoREksSUFBSSxFQUFFVTtRQUNSLENBQUMsQ0FBQyxHQUNBLE1BQU1kLGVBQWUsQ0FBQyw2Q0FBNkMsRUFBRTtVQUNyRSxHQUFHMEMsYUFBYTtVQUNoQnRDLElBQUksRUFBRVU7UUFDUixDQUFDLENBQUM7UUFDSixPQUFPbEIsNEJBQTRCLENBQUNPLEVBQUUsQ0FBQztVQUNyQ0MsSUFBSSxFQUFFO1lBQ0pELEVBQUUsRUFBRSxJQUFJO1lBQ1JELFFBQVEsRUFBRUE7VUFDWjtRQUNGLENBQUMsQ0FBQztNQUNKLENBQUMsQ0FBQyxPQUFPRyxHQUFHLEVBQUU7UUFDWkMsT0FBTyxDQUFDQyxHQUFHLENBQUMsNkNBQTZDLEVBQUVGLEdBQUcsQ0FBQztRQUMvRCxJQUFJLElBQUE2QyxxQ0FBb0IsRUFBQzdDLEdBQUcsQ0FBQyxFQUFFO1VBQzdCLE9BQU9ULDRCQUE0QixDQUFDTyxFQUFFLENBQUM7WUFDckNDLElBQUksRUFBRTtjQUFFRCxFQUFFLEVBQUUsSUFBSTtjQUFFRCxRQUFRLEVBQUU7Z0JBQUU0QyxnQkFBZ0IsRUFBRSxDQUFDO2dCQUFFQyxXQUFXLEVBQUU7Y0FBRztZQUFFO1VBQ3ZFLENBQUMsQ0FBQztRQUNKO1FBQ0EsT0FBT25ELDRCQUE0QixDQUFDTyxFQUFFLENBQUM7VUFDckNDLElBQUksRUFBRTtZQUNKRCxFQUFFLEVBQUUsS0FBSztZQUNUSyxLQUFLLEVBQUUsSUFBQUMsZ0NBQWUsRUFBQ0osR0FBRztVQUM1QjtRQUNGLENBQUMsQ0FBQztNQUNKO0lBQ0YsQ0FBQztJQXAzQ0MsSUFBSSxDQUFDYixNQUFNLEdBQUdBLE1BQU07SUFDcEIsSUFBSSxDQUFDQyxpQkFBaUIsR0FBR0EsaUJBQWlCO0VBQzVDO0FBbTNDRjtBQUFDeU0sT0FBQSxDQUFBQyxPQUFBLEdBQUE3TSxlQUFBIn0=