<?php

declare(strict_types=1);
/**
 * SPDX-FileCopyrightText: 2018 Nextcloud GmbH and Nextcloud contributors
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */
namespace OCA\Talk\Migration;

use OCP\DB\ISchemaWrapper;
use OCP\DB\QueryBuilder\IQueryBuilder;
use OCP\DB\Types;
use OCP\IDBConnection;
use OCP\Migration\IOutput;
use OCP\Migration\SimpleMigrationStep;

class Version3003Date20180718112436 extends SimpleMigrationStep {

	public function __construct(
		protected IDBConnection $connection,
	) {
	}

	/**
	 * @param IOutput $output
	 * @param \Closure $schemaClosure The `\Closure` returns a `ISchemaWrapper`
	 * @param array $options
	 * @return null|ISchemaWrapper
	 * @since 13.0.0
	 */
	#[\Override]
	public function changeSchema(IOutput $output, \Closure $schemaClosure, array $options): ?ISchemaWrapper {
		/** @var ISchemaWrapper $schema */
		$schema = $schemaClosure();

		$table = $schema->getTable('talk_rooms');
		if (!$table->hasColumn('last_activity')) {
			$table->addColumn('last_activity', Types::DATETIME, [
				'notnull' => false,
			]);

			$table->addIndex(['last_activity'], 'last_active');
		}

		return $schema;
	}

	/**
	 * @param IOutput $output
	 * @param \Closure $schemaClosure The `\Closure` returns a `ISchemaWrapper`
	 * @param array $options
	 * @since 13.0.0
	 */
	#[\Override]
	public function postSchemaChange(IOutput $output, \Closure $schemaClosure, array $options): void {
		$update = $this->connection->getQueryBuilder();
		$update->update('talk_rooms')
			->set('last_activity', $update->createParameter('activity'))
			->where($update->expr()->eq('id', $update->createParameter('room')));

		$query = $this->connection->getQueryBuilder();
		$query->select('object_id')
			->selectAlias($query->func()->max('creation_timestamp'), 'last_activity')
			->from('comments')
			->where($query->expr()->eq('object_type', $query->createNamedParameter('chat')))
			->groupBy('object_id');

		$result = $query->executeQuery();
		while ($row = $result->fetch()) {
			$lastActivity = new \DateTime($row['last_activity']);
			$update->setParameter('activity', $lastActivity, IQueryBuilder::PARAM_DATE)
				->setParameter('room', $row['object_id']);
			$update->executeStatement();
		}
		$result->closeCursor();
	}
}
