/*
 * Decompiled with CFR 0.152.
 */
package ghidra.program.database.mem;

import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.mem.Memory;
import ghidra.program.model.mem.MemoryAccessException;
import ghidra.program.model.mem.MemoryBlock;

public class ByteMappingScheme {
    private final int mappedByteCount;
    private final int nonMappedByteCount;
    private final int mappedSourceByteCount;

    ByteMappingScheme(int encodedMappingScheme) throws IllegalArgumentException {
        if (encodedMappingScheme == 0) {
            this.mappedByteCount = 1;
            this.mappedSourceByteCount = 1;
            this.nonMappedByteCount = 0;
        } else {
            this.mappedByteCount = ByteMappingScheme.getMappedByteCount(encodedMappingScheme);
            this.mappedSourceByteCount = ByteMappingScheme.getMappedSourceByteCount(encodedMappingScheme);
            this.nonMappedByteCount = this.mappedSourceByteCount - this.mappedByteCount;
            ByteMappingScheme.validateMappingScheme(this.mappedByteCount, this.mappedSourceByteCount);
        }
    }

    public ByteMappingScheme(int mappedByteCount, int mappedSourceByteCount) {
        ByteMappingScheme.validateMappingScheme(mappedByteCount, mappedSourceByteCount);
        this.mappedByteCount = mappedByteCount;
        this.mappedSourceByteCount = mappedSourceByteCount;
        this.nonMappedByteCount = mappedSourceByteCount - mappedByteCount;
    }

    public ByteMappingScheme(String mappingScheme) {
        int index = mappingScheme.indexOf(58);
        if (index < 0) {
            throw new IllegalArgumentException("invalid mapping scheme: " + mappingScheme);
        }
        String mappedByteCountStr = mappingScheme.substring(0, index);
        String sourceByteCountStr = mappingScheme.substring(index + 1);
        try {
            this.mappedByteCount = Integer.parseInt(mappedByteCountStr);
            this.mappedSourceByteCount = Integer.parseInt(sourceByteCountStr);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("invalid mapping scheme: " + mappingScheme);
        }
        ByteMappingScheme.validateMappingScheme(this.mappedByteCount, this.mappedSourceByteCount);
        this.nonMappedByteCount = this.mappedSourceByteCount - this.mappedByteCount;
    }

    public String toString() {
        Object ratioStr = "1:1";
        if (!this.isOneToOneMapping()) {
            ratioStr = this.mappedByteCount + ":" + this.mappedSourceByteCount;
        }
        return (String)ratioStr + " mapping";
    }

    int getEncodedMappingScheme() {
        if (this.isOneToOneMapping()) {
            return 0;
        }
        return ByteMappingScheme.getEncodedMappingScheme(this.mappedByteCount, this.mappedSourceByteCount);
    }

    public boolean isOneToOneMapping() {
        return this.mappedSourceByteCount <= 1;
    }

    public int getMappedByteCount() {
        if (this.isOneToOneMapping()) {
            return 1;
        }
        return this.mappedByteCount;
    }

    public int getMappedSourceByteCount() {
        if (this.isOneToOneMapping()) {
            return 1;
        }
        return this.mappedSourceByteCount;
    }

    public Address getMappedSourceAddress(Address mappedSourceBaseAddress, long offsetInSubBlock) throws AddressOverflowException {
        if (offsetInSubBlock < 0L) {
            throw new IllegalArgumentException("negative offset");
        }
        long sourceOffset = offsetInSubBlock;
        if (!this.isOneToOneMapping()) {
            sourceOffset = (long)this.mappedSourceByteCount * (offsetInSubBlock / (long)this.mappedByteCount) + offsetInSubBlock % (long)this.mappedByteCount;
        }
        return mappedSourceBaseAddress.addNoWrap(sourceOffset);
    }

    Address getMappedAddress(MemoryBlock mappedBlock, long mappedSourceOffset, boolean skipBack) throws AddressOverflowException {
        if (mappedSourceOffset < 0L) {
            throw new IllegalArgumentException("negative source offset");
        }
        long mappedOffset = mappedSourceOffset;
        if (!this.isOneToOneMapping()) {
            mappedOffset = (long)this.mappedByteCount * (mappedSourceOffset / (long)this.mappedSourceByteCount);
            long offsetLimit = mappedBlock.getSize() - 1L;
            long mod = mappedSourceOffset % (long)this.mappedSourceByteCount;
            if (mod < (long)this.mappedByteCount) {
                mappedOffset += mod;
            } else if (!skipBack && (mappedOffset += (long)this.mappedByteCount) > offsetLimit) {
                return null;
            }
        }
        return mappedBlock.getStart().addNoWrap(mappedOffset);
    }

    int getBytes(Memory memory, Address mappedSourceBaseAddress, long offsetInSubBlock, byte[] b, int off, int len) throws MemoryAccessException, AddressOverflowException {
        if (this.isOneToOneMapping()) {
            return memory.getBytes(mappedSourceBaseAddress.addNoWrap(offsetInSubBlock), b, off, len);
        }
        long patternCount = offsetInSubBlock / (long)this.mappedByteCount;
        int partialByteCount = (int)(offsetInSubBlock % (long)this.mappedByteCount);
        long mappedOffset = (long)this.mappedSourceByteCount * patternCount + (long)partialByteCount;
        int bufSize = this.mappedSourceByteCount * (len / this.mappedByteCount + 1);
        byte[] buf = new byte[bufSize];
        int bufCnt = memory.getBytes(mappedSourceBaseAddress.addNoWrap(mappedOffset), buf);
        int cnt = 0;
        int index = off;
        int i = this.mappedByteCount - partialByteCount;
        boolean skip = false;
        for (int bufIndex = 0; bufIndex < bufCnt && cnt < len; ++bufIndex) {
            if (!skip) {
                b[index++] = buf[bufIndex];
                ++cnt;
                if (--i != 0) continue;
                skip = true;
                i = this.nonMappedByteCount;
                continue;
            }
            if (--i != 0) continue;
            skip = false;
            i = this.mappedByteCount;
        }
        return cnt;
    }

    void setBytes(Memory memory, Address mappedSourceBaseAddress, long offsetInSubBlock, byte[] b, int off, int len) throws MemoryAccessException, AddressOverflowException {
        if (this.isOneToOneMapping()) {
            memory.setBytes(mappedSourceBaseAddress.addNoWrap(offsetInSubBlock), b, off, len);
            return;
        }
        long patternCount = offsetInSubBlock / (long)this.mappedByteCount;
        int partialByteCount = (int)(offsetInSubBlock % (long)this.mappedByteCount);
        long mappedOffset = (long)this.mappedSourceByteCount * patternCount + (long)partialByteCount;
        Address destAddr = mappedSourceBaseAddress.addNoWrap(mappedOffset);
        int index = off;
        int cnt = 0;
        int i = this.mappedByteCount - partialByteCount;
        while (cnt < len) {
            memory.setBytes(destAddr, b, index, i);
            index += i;
            cnt += i;
            destAddr = destAddr.addNoWrap(i + this.nonMappedByteCount);
            i = this.mappedByteCount;
        }
    }

    static void validateMappingScheme(int schemeDestByteCount, int schemeSrcByteCount) {
        if (schemeDestByteCount <= 0 || schemeDestByteCount > 127 || schemeSrcByteCount <= 0 || schemeSrcByteCount > 127 || schemeDestByteCount > schemeSrcByteCount) {
            throw new IllegalArgumentException("invalid byte mapping ratio: " + schemeDestByteCount + ":" + schemeSrcByteCount);
        }
    }

    static int getEncodedMappingScheme(int schemeDestByteCount, int schemeSrcByteCount) {
        ByteMappingScheme.validateMappingScheme(schemeDestByteCount, schemeSrcByteCount);
        return schemeDestByteCount << 7 | schemeSrcByteCount & 0x7F;
    }

    static int getMappedByteCount(int mappingScheme) {
        int mappedByteCount = 1;
        if (mappingScheme != 0) {
            mappedByteCount = mappingScheme >> 7 & 0x7F;
        }
        return mappedByteCount;
    }

    static int getMappedSourceByteCount(int mappingScheme) {
        int mappedSourceByteCount = 1;
        if (mappingScheme != 0) {
            mappedSourceByteCount = mappingScheme & 0x7F;
        }
        return mappedSourceByteCount;
    }
}

