﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/clouddirectory/CloudDirectoryRequest.h>
#include <aws/clouddirectory/CloudDirectory_EXPORTS.h>
#include <aws/clouddirectory/model/ConsistencyLevel.h>
#include <aws/clouddirectory/model/ObjectReference.h>
#include <aws/clouddirectory/model/SchemaFacet.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CloudDirectory {
namespace Model {

/**
 */
class GetObjectAttributesRequest : public CloudDirectoryRequest {
 public:
  AWS_CLOUDDIRECTORY_API GetObjectAttributesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetObjectAttributes"; }

  AWS_CLOUDDIRECTORY_API Aws::String SerializePayload() const override;

  AWS_CLOUDDIRECTORY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that is associated with the <a>Directory</a>
   * where the object resides.</p>
   */
  inline const Aws::String& GetDirectoryArn() const { return m_directoryArn; }
  inline bool DirectoryArnHasBeenSet() const { return m_directoryArnHasBeenSet; }
  template <typename DirectoryArnT = Aws::String>
  void SetDirectoryArn(DirectoryArnT&& value) {
    m_directoryArnHasBeenSet = true;
    m_directoryArn = std::forward<DirectoryArnT>(value);
  }
  template <typename DirectoryArnT = Aws::String>
  GetObjectAttributesRequest& WithDirectoryArn(DirectoryArnT&& value) {
    SetDirectoryArn(std::forward<DirectoryArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Reference that identifies the object whose attributes will be retrieved.</p>
   */
  inline const ObjectReference& GetObjectReference() const { return m_objectReference; }
  inline bool ObjectReferenceHasBeenSet() const { return m_objectReferenceHasBeenSet; }
  template <typename ObjectReferenceT = ObjectReference>
  void SetObjectReference(ObjectReferenceT&& value) {
    m_objectReferenceHasBeenSet = true;
    m_objectReference = std::forward<ObjectReferenceT>(value);
  }
  template <typename ObjectReferenceT = ObjectReference>
  GetObjectAttributesRequest& WithObjectReference(ObjectReferenceT&& value) {
    SetObjectReference(std::forward<ObjectReferenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The consistency level at which to retrieve the attributes on an object.</p>
   */
  inline ConsistencyLevel GetConsistencyLevel() const { return m_consistencyLevel; }
  inline bool ConsistencyLevelHasBeenSet() const { return m_consistencyLevelHasBeenSet; }
  inline void SetConsistencyLevel(ConsistencyLevel value) {
    m_consistencyLevelHasBeenSet = true;
    m_consistencyLevel = value;
  }
  inline GetObjectAttributesRequest& WithConsistencyLevel(ConsistencyLevel value) {
    SetConsistencyLevel(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Identifier for the facet whose attributes will be retrieved. See
   * <a>SchemaFacet</a> for details.</p>
   */
  inline const SchemaFacet& GetSchemaFacet() const { return m_schemaFacet; }
  inline bool SchemaFacetHasBeenSet() const { return m_schemaFacetHasBeenSet; }
  template <typename SchemaFacetT = SchemaFacet>
  void SetSchemaFacet(SchemaFacetT&& value) {
    m_schemaFacetHasBeenSet = true;
    m_schemaFacet = std::forward<SchemaFacetT>(value);
  }
  template <typename SchemaFacetT = SchemaFacet>
  GetObjectAttributesRequest& WithSchemaFacet(SchemaFacetT&& value) {
    SetSchemaFacet(std::forward<SchemaFacetT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of attribute names whose values will be retrieved.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAttributeNames() const { return m_attributeNames; }
  inline bool AttributeNamesHasBeenSet() const { return m_attributeNamesHasBeenSet; }
  template <typename AttributeNamesT = Aws::Vector<Aws::String>>
  void SetAttributeNames(AttributeNamesT&& value) {
    m_attributeNamesHasBeenSet = true;
    m_attributeNames = std::forward<AttributeNamesT>(value);
  }
  template <typename AttributeNamesT = Aws::Vector<Aws::String>>
  GetObjectAttributesRequest& WithAttributeNames(AttributeNamesT&& value) {
    SetAttributeNames(std::forward<AttributeNamesT>(value));
    return *this;
  }
  template <typename AttributeNamesT = Aws::String>
  GetObjectAttributesRequest& AddAttributeNames(AttributeNamesT&& value) {
    m_attributeNamesHasBeenSet = true;
    m_attributeNames.emplace_back(std::forward<AttributeNamesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryArn;

  ObjectReference m_objectReference;

  ConsistencyLevel m_consistencyLevel{ConsistencyLevel::NOT_SET};

  SchemaFacet m_schemaFacet;

  Aws::Vector<Aws::String> m_attributeNames;
  bool m_directoryArnHasBeenSet = false;
  bool m_objectReferenceHasBeenSet = false;
  bool m_consistencyLevelHasBeenSet = false;
  bool m_schemaFacetHasBeenSet = false;
  bool m_attributeNamesHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudDirectory
}  // namespace Aws
